<?php
session_start();
require_once 'config/config.php';
require_once 'includes/functions.php';

$token = $_GET['token'] ?? '';

if (!$token) {
    redirect('index.php');
}

try {
    // Teklifi token ile al
    $stmt = $pdo->prepare("
        SELECT q.*, qr.additional_notes as request_notes,
               c.first_name, c.last_name, c.phone, c.email,
               v.brand, v.model, v.year, v.fuel_type, v.transmission, v.plate
        FROM quotes q
        JOIN quote_requests qr ON q.quote_request_id = qr.id
        JOIN customers c ON qr.customer_id = c.id
        JOIN vehicles v ON qr.vehicle_id = v.id
        WHERE q.quote_token = ? AND q.valid_until > NOW()
    ");
    $stmt->execute([$token]);
    $quote = $stmt->fetch();
    
    if (!$quote) {
        $error_message = 'Teklif bulunamadı veya süresi dolmuş.';
    } else {
        // Teklif kalemlerini al
        $stmt = $pdo->prepare("
            SELECT * FROM quote_items 
            WHERE quote_id = ? 
            ORDER BY sort_order, id
        ");
        $stmt->execute([$quote['id']]);
        $quote_items = $stmt->fetchAll();
        
        // Görüntülenme kaydı ve durum güncelleme
        if (!$quote['viewed_at']) {
            $stmt = $pdo->prepare("UPDATE quotes SET viewed_at = NOW(), status = 'VIEWED' WHERE id = ?");
            $stmt->execute([$quote['id']]);
        }
    }
    
} catch (Exception $e) {
    error_log("View quote error: " . $e->getMessage());
    $error_message = 'Bir hata oluştu.';
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Teklif Görüntüle - Araç Bakım Servisi</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        :root {
            --primary-blue: #2563eb;
            --secondary-blue: #3b82f6;
            --light-blue: #dbeafe;
            --dark-blue: #1e40af;
            --pure-white: #ffffff;
            --light-gray: #f8fafc;
            --medium-gray: #64748b;
            --dark-gray: #334155;
            --success-green: #10b981;
            --light-green: #d1fae5;
            --gradient-primary: linear-gradient(135deg, #2563eb 0%, #3b82f6 100%);
            --gradient-light: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            --shadow-sm: 0 1px 2px 0 rgb(0 0 0 / 0.05);
            --shadow-md: 0 4px 6px -1px rgb(0 0 0 / 0.1), 0 2px 4px -2px rgb(0 0 0 / 0.1);
            --shadow-lg: 0 10px 15px -3px rgb(0 0 0 / 0.1), 0 4px 6px -4px rgb(0 0 0 / 0.1);
            --shadow-xl: 0 20px 25px -5px rgb(0 0 0 / 0.1), 0 8px 10px -6px rgb(0 0 0 / 0.1);
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Inter', sans-serif;
            line-height: 1.6;
            color: var(--dark-gray);
            background: var(--gradient-light);
            min-height: 100vh;
        }

        /* Modern Navigation */
        .modern-navbar {
            background: var(--pure-white);
            box-shadow: var(--shadow-md);
            padding: 1rem 0;
        }

        .navbar-brand {
            font-family: 'Poppins', sans-serif;
            font-weight: 700;
            font-size: 1.5rem;
            color: var(--primary-blue) !important;
            text-decoration: none;
        }

        .nav-link {
            color: var(--medium-gray) !important;
            font-weight: 500;
            text-decoration: none;
            transition: color 0.3s ease;
        }

        .nav-link:hover {
            color: var(--primary-blue) !important;
        }

        /* Quote Container */
        .quote-container {
            max-width: 1200px;
            margin: 2rem auto;
            padding: 0 1rem;
        }

        .quote-card {
            background: var(--pure-white);
            border-radius: 20px;
            box-shadow: var(--shadow-xl);
            overflow: hidden;
            border: 1px solid rgba(37, 99, 235, 0.1);
        }

        .quote-header {
            background: var(--gradient-primary);
            color: white;
            padding: 2rem;
            text-align: center;
            position: relative;
        }

        .quote-header::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><defs><pattern id="grid" width="10" height="10" patternUnits="userSpaceOnUse"><path d="M 10 0 L 0 0 0 10" fill="none" stroke="%23ffffff" stroke-width="0.5" opacity="0.1"/></pattern></defs><rect width="100" height="100" fill="url(%23grid)"/></svg>');
        }

        .quote-title {
            font-family: 'Poppins', sans-serif;
            font-size: 2rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
            position: relative;
            z-index: 2;
        }

        .quote-subtitle {
            font-size: 1rem;
            opacity: 0.9;
            position: relative;
            z-index: 2;
        }

        .quote-body {
            padding: 2rem;
        }

        /* Quote Info */
        .quote-info {
            background: var(--light-gray);
            border-radius: 16px;
            padding: 1.5rem;
            margin-bottom: 2rem;
            border: 1px solid rgba(37, 99, 235, 0.1);
        }

        .info-title {
            font-family: 'Poppins', sans-serif;
            font-size: 1.25rem;
            font-weight: 600;
            color: var(--dark-blue);
            margin-bottom: 1rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .info-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
        }

        .info-item {
            display: flex;
            flex-direction: column;
        }

        .info-label {
            font-size: 0.875rem;
            color: var(--medium-gray);
            font-weight: 500;
            margin-bottom: 0.25rem;
        }

        .info-value {
            font-weight: 600;
            color: var(--dark-gray);
        }

        /* Quote Items */
        .quote-items {
            margin-bottom: 2rem;
        }

        .item-card {
            background: var(--pure-white);
            border: 2px solid #e2e8f0;
            border-radius: 12px;
            padding: 1.5rem;
            margin-bottom: 1rem;
            transition: all 0.3s ease;
            cursor: pointer;
        }

        .item-card:hover {
            border-color: var(--primary-blue);
            transform: translateY(-2px);
            box-shadow: var(--shadow-md);
        }

        .item-card.selected {
            border-color: var(--primary-blue);
            background: var(--light-blue);
        }

        .item-header {
            display: flex;
            justify-content: between;
            align-items: flex-start;
            margin-bottom: 1rem;
        }

        .item-name {
            font-family: 'Poppins', sans-serif;
            font-size: 1.125rem;
            font-weight: 600;
            color: var(--dark-blue);
            margin-bottom: 0.5rem;
        }

        .item-description {
            color: var(--medium-gray);
            font-size: 0.875rem;
            margin-bottom: 1rem;
        }

        .item-controls {
            display: flex;
            justify-content: between;
            align-items: center;
            gap: 1rem;
        }

        .quantity-control {
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .quantity-btn {
            width: 32px;
            height: 32px;
            border: 2px solid var(--primary-blue);
            background: transparent;
            color: var(--primary-blue);
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .quantity-btn:hover {
            background: var(--primary-blue);
            color: white;
        }

        .quantity-input {
            width: 60px;
            text-align: center;
            border: 2px solid #e2e8f0;
            border-radius: 8px;
            padding: 0.5rem;
            font-weight: 600;
        }

        .item-price {
            font-family: 'Poppins', sans-serif;
            font-size: 1.25rem;
            font-weight: 700;
            color: var(--primary-blue);
        }

        /* Cart Summary */
        .cart-summary {
            background: var(--light-blue);
            border-radius: 16px;
            padding: 2rem;
            margin-bottom: 2rem;
            border: 2px solid rgba(37, 99, 235, 0.2);
            position: sticky;
            top: 2rem;
        }

        .summary-title {
            font-family: 'Poppins', sans-serif;
            font-size: 1.5rem;
            font-weight: 700;
            color: var(--dark-blue);
            margin-bottom: 1.5rem;
            text-align: center;
        }

        .summary-item {
            display: flex;
            justify-content: between;
            align-items: center;
            padding: 0.75rem 0;
            border-bottom: 1px solid rgba(37, 99, 235, 0.2);
        }

        .summary-item:last-child {
            border-bottom: none;
            font-size: 1.25rem;
            font-weight: 700;
            color: var(--dark-blue);
        }

        /* Buttons */
        .modern-btn {
            background: var(--gradient-primary);
            color: white;
            border: none;
            padding: 0.875rem 2rem;
            border-radius: 12px;
            font-weight: 600;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            transition: all 0.3s ease;
            box-shadow: var(--shadow-md);
            cursor: pointer;
            width: 100%;
            justify-content: center;
        }

        .modern-btn:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
            color: white;
        }

        .modern-btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none;
        }

        .modern-btn-outline {
            background: transparent;
            color: var(--primary-blue);
            border: 2px solid var(--primary-blue);
            padding: 0.875rem 2rem;
            border-radius: 12px;
            font-weight: 600;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            transition: all 0.3s ease;
            cursor: pointer;
            width: 100%;
            justify-content: center;
        }

        .modern-btn-outline:hover {
            background: var(--primary-blue);
            color: white;
            transform: translateY(-2px);
            box-shadow: var(--shadow-md);
        }

        /* Error Message */
        .error-card {
            background: #fef2f2;
            border: 2px solid #fecaca;
            border-radius: 16px;
            padding: 2rem;
            text-align: center;
            color: #dc2626;
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .quote-container {
                margin: 1rem auto;
                padding: 0 0.5rem;
            }

            .quote-body {
                padding: 1.5rem;
            }

            .info-grid {
                grid-template-columns: 1fr;
            }

            .item-controls {
                flex-direction: column;
                align-items: stretch;
                gap: 1rem;
            }

            .cart-summary {
                position: static;
            }
        }

        /* Animations */
        @keyframes fadeInUp {
            from {
                opacity: 0;
                transform: translateY(30px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .animate-fade-in-up {
            animation: fadeInUp 0.6s ease-out;
        }
    </style>
</head>
<body>
    <!-- Modern Navigation -->
    <nav class="navbar modern-navbar">
        <div class="container">
            <a class="navbar-brand" href="index.php">
                <i class="fas fa-car me-2"></i>
                Araç Bakım Servisi
            </a>
            <div class="navbar-nav">
                <a class="nav-link" href="index.php">
                    <i class="fas fa-home me-2"></i>Ana Sayfa
                </a>
            </div>
        </div>
    </nav>

    <div class="quote-container">
        <?php if (isset($error_message)): ?>
        <div class="error-card animate-fade-in-up">
            <i class="fas fa-exclamation-triangle fa-3x mb-3"></i>
            <h3>Teklif Bulunamadı</h3>
            <p><?= h($error_message) ?></p>
            <a href="index.php" class="modern-btn-outline mt-3" style="width: auto;">
                <i class="fas fa-home"></i> Ana Sayfaya Dön
            </a>
        </div>
        <?php else: ?>
        <div class="quote-card animate-fade-in-up">
            <div class="quote-header">
                <h1 class="quote-title">
                    <i class="fas fa-file-invoice-dollar me-2"></i>
                    Teklif Detayları
                </h1>
                <p class="quote-subtitle">
                    Aşağıdaki hizmetlerden istediğinizi seçerek sepetinizi oluşturun
                </p>
            </div>

            <div class="quote-body">
                <div class="row">
                    <div class="col-lg-8">
                        <!-- Quote Info -->
                        <div class="quote-info">
                            <h3 class="info-title">
                                <i class="fas fa-info-circle"></i>
                                Teklif Bilgileri
                            </h3>
                            <div class="info-grid">
                                <div class="info-item">
                                    <span class="info-label">Teklif Numarası</span>
                                    <span class="info-value"><?= h($quote['quote_number'] ?? 'TKL-' . $quote['id']) ?></span>
                                </div>
                                <div class="info-item">
                                    <span class="info-label">Müşteri</span>
                                    <span class="info-value"><?= h(($quote['first_name'] ?? '') . ' ' . ($quote['last_name'] ?? '')) ?></span>
                                </div>
                                <div class="info-item">
                                    <span class="info-label">Araç</span>
                                    <span class="info-value"><?= h(($quote['brand'] ?? '') . ' ' . ($quote['model'] ?? '') . ' (' . ($quote['year'] ?? '') . ')') ?></span>
                                </div>
                                <div class="info-item">
                                    <span class="info-label">Geçerlilik Tarihi</span>
                                    <span class="info-value"><?= formatDate($quote['valid_until'], 'd.m.Y H:i') ?></span>
                                </div>
                            </div>
                        </div>

                        <!-- Quote Items -->
                        <div class="quote-items">
                            <h3 style="color: var(--dark-blue); font-weight: 600; margin-bottom: 1.5rem;">
                                <i class="fas fa-list me-2"></i>PARÇA SEÇİMİ
                            </h3>
                            
                            <?php foreach ($quote_items as $item): ?>
                            <div class="item-card" data-item-id="<?= $item['id'] ?>" onclick="toggleItem(<?= $item['id'] ?>)">
                                <div class="item-header">
                                    <div style="flex: 1;">
                                        <h4 class="item-name"><?= h($item['description'] ?? 'Ürün Adı Yok') ?></h4>
                                    </div>
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="item_<?= $item['id'] ?>" 
                                               onchange="updateSelection(<?= $item['id'] ?>)">
                                    </div>
                                </div>
                                
                                <div class="item-controls">
                                    <div class="quantity-control">
                                        <button type="button" class="quantity-btn" onclick="changeQuantity(<?= $item['id'] ?>, -1)">
                                            <i class="fas fa-minus"></i>
                                        </button>
                                        <input type="number" class="quantity-input" id="quantity_<?= $item['id'] ?>" 
                                               value="1" min="1" onchange="updateQuantity(<?= $item['id'] ?>)">
                                        <button type="button" class="quantity-btn" onclick="changeQuantity(<?= $item['id'] ?>, 1)">
                                            <i class="fas fa-plus"></i>
                                        </button>
                                    </div>
                                    <div class="item-price">
                                        <?= formatMoney($item['unit_price']) ?>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>

                        <?php if ($quote['notes']): ?>
                        <div class="quote-info">
                            <h3 class="info-title">
                                <i class="fas fa-sticky-note"></i>
                                Teklif Notları
                            </h3>
                            <p style="color: var(--dark-gray); line-height: 1.6;">
                                <?= nl2br(h($quote['notes'] ?? 'Ek not bulunmuyor.')) ?>
                            </p>
                        </div>
                        <?php endif; ?>
                    </div>

                    <div class="col-lg-4">
                        <!-- Cart Summary -->
                        <div class="cart-summary">
                            <h3 class="summary-title">
                                <i class="fas fa-shopping-cart me-2"></i>
                                Sepet Özeti
                            </h3>
                            
                            <div id="cart-items">
                                <div class="text-center text-muted py-4">
                                    <i class="fas fa-shopping-cart fa-2x mb-2"></i>
                                    <p>Henüz ürün seçilmedi</p>
                                </div>
                            </div>
                            
                            <div id="cart-summary" style="display: none;">
                                <div class="summary-item">
                                    <span>Ara Toplam:</span>
                                    <span id="subtotal">₺0.00</span>
                                </div>
                                <div class="summary-item">
                                    <span>KDV (<?= $quote['tax_rate'] ?>%):</span>
                                    <span id="tax-amount">₺0.00</span>
                                </div>
                                <div class="summary-item">
                                    <span>Toplam:</span>
                                    <span id="total-amount">₺0.00</span>
                                </div>
                            </div>
                            
                            <div class="mt-3">
                                <button type="button" class="modern-btn" id="checkout-btn" 
                                        onclick="proceedToPayment()" disabled>
                                    <i class="fas fa-credit-card"></i>
                                    Ödemeye Geç
                                </button>
                                
                                <button type="button" class="modern-btn-outline mt-2" onclick="clearCart()">
                                    <i class="fas fa-trash"></i>
                                    Sepeti Temizle
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        let selectedItems = {};
        const quoteItems = <?= json_encode($quote_items ?? []) ?>;
        const taxRate = <?= $quote['tax_rate'] ?? 0 ?>;

        function toggleItem(itemId) {
            const checkbox = document.getElementById(`item_${itemId}`);
            checkbox.checked = !checkbox.checked;
            updateSelection(itemId);
        }

        function updateSelection(itemId) {
            const checkbox = document.getElementById(`item_${itemId}`);
            const card = document.querySelector(`[data-item-id="${itemId}"]`);
            const quantity = parseInt(document.getElementById(`quantity_${itemId}`).value);
            
            if (checkbox.checked) {
                selectedItems[itemId] = quantity;
                card.classList.add('selected');
            } else {
                delete selectedItems[itemId];
                card.classList.remove('selected');
            }
            
            updateCartDisplay();
        }

        function changeQuantity(itemId, change) {
            const input = document.getElementById(`quantity_${itemId}`);
            const newValue = Math.max(1, parseInt(input.value) + change);
            input.value = newValue;
            updateQuantity(itemId);
        }

        function updateQuantity(itemId) {
            const quantity = parseInt(document.getElementById(`quantity_${itemId}`).value);
            if (selectedItems[itemId]) {
                selectedItems[itemId] = quantity;
                updateCartDisplay();
            }
        }

        function updateCartDisplay() {
            const cartItemsDiv = document.getElementById('cart-items');
            const cartSummaryDiv = document.getElementById('cart-summary');
            const checkoutBtn = document.getElementById('checkout-btn');
            
            if (Object.keys(selectedItems).length === 0) {
                cartItemsDiv.innerHTML = `
                    <div class="text-center text-muted py-4">
                        <i class="fas fa-shopping-cart fa-2x mb-2"></i>
                        <p>Henüz ürün seçilmedi</p>
                    </div>
                `;
                cartSummaryDiv.style.display = 'none';
                checkoutBtn.disabled = true;
                return;
            }
            
            let html = '';
            let subtotal = 0;
            
            Object.keys(selectedItems).forEach(itemId => {
                const item = quoteItems.find(i => i.id == itemId);
                const quantity = selectedItems[itemId];
                const total = item.unit_price * quantity;
                subtotal += total;
                
                html += `
                    <div class="d-flex justify-content-between align-items-center mb-2 pb-2 border-bottom">
                        <div>
                            <small class="fw-bold">${item.description || 'Ürün Adı Yok'}</small>
                            <br><small class="text-muted">x${quantity}</small>
                        </div>
                        <small class="fw-bold">${formatMoney(total)}</small>
                    </div>
                `;
            });
            
            const taxAmount = subtotal * (taxRate / 100);
            const total = subtotal + taxAmount;
            
            cartItemsDiv.innerHTML = html;
            cartSummaryDiv.style.display = 'block';
            
            document.getElementById('subtotal').textContent = formatMoney(subtotal);
            document.getElementById('tax-amount').textContent = formatMoney(taxAmount);
            document.getElementById('total-amount').textContent = formatMoney(total);
            
            checkoutBtn.disabled = false;
        }

        function clearCart() {
            selectedItems = {};
            document.querySelectorAll('.form-check-input').forEach(cb => cb.checked = false);
            document.querySelectorAll('.item-card').forEach(card => card.classList.remove('selected'));
            updateCartDisplay();
        }

        function proceedToPayment() {
            if (Object.keys(selectedItems).length === 0) {
                alert('Lütfen en az bir hizmet seçin.');
                return;
            }
            
            // Seçili ürünleri sunucuya gönder
            fetch('update_cart.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    token: '<?= $token ?>',
                    selected_items: selectedItems
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    window.location.href = `payment.php?token=<?= $token ?>`;
                } else {
                    alert('Hata: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Bir hata oluştu.');
            });
        }

        function formatMoney(amount) {
            return new Intl.NumberFormat('tr-TR', {
                style: 'currency',
                currency: 'TRY'
            }).format(amount);
        }

        // Sayfa yüklendiğinde
        document.addEventListener('DOMContentLoaded', function() {
            updateCartDisplay();
        });
    </script>
</body>
</html>