<?php
session_start();
require_once 'config/config.php';
require_once 'includes/functions.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Geçersiz istek metodu.']);
    exit;
}

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    $quote_token = $input['token'] ?? '';
    $selected_items = $input['selected_items'] ?? [];
    
    if (empty($quote_token)) {
        throw new Exception('Teklif token\'ı gereklidir.');
    }
    
    if (empty($selected_items)) {
        throw new Exception('Seçilen ürün bilgisi gereklidir.');
    }
    
    // Teklifi doğrula
    $stmt = $pdo->prepare("
        SELECT q.*, qr.customer_id, qr.vehicle_id 
        FROM quotes q
        JOIN quote_requests qr ON q.quote_request_id = qr.id
        WHERE q.quote_token = ? AND q.valid_until > NOW()
    ");
    $stmt->execute([$quote_token]);
    $quote = $stmt->fetch();
    
    if (!$quote) {
        throw new Exception('Teklif bulunamadı veya süresi dolmuş.');
    }
    
    // Teklif kalemlerini al
    $item_ids = array_keys($selected_items);
    $placeholders = str_repeat('?,', count($item_ids) - 1) . '?';
    
    $stmt = $pdo->prepare("
        SELECT * FROM quote_items 
        WHERE quote_id = ? AND id IN ($placeholders)
        ORDER BY sort_order
    ");
    $stmt->execute(array_merge([$quote['id']], $item_ids));
    $quote_items = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Seçilen ürünleri session'a kaydet
    $cart_items = [];
    $subtotal = 0;
    
    foreach ($quote_items as $quote_item) {
        $item_id = $quote_item['id'];
        $quantity = intval($selected_items[$item_id] ?? 1);
        
        if ($quantity > 0) {
            $line_total = $quote_item['unit_price'] * $quantity;
            
            $cart_items[$item_id] = [
                'quote_item_id' => $item_id,
                'name' => $quote_item['description'] ?? 'Ürün Adı Yok',
                'description' => $quote_item['description'],
                'quantity' => $quantity,
                'unit_price' => $quote_item['unit_price'],
                'line_total' => $line_total
            ];
            
            $subtotal += $line_total;
        }
    }
    
    if (empty($cart_items)) {
        throw new Exception('Geçerli ürün seçimi bulunamadı.');
    }
    
    // İndirim ve vergi hesapla
    $discount_rate = floatval($quote['discount_rate'] ?? 0);
    $tax_rate = floatval($quote['tax_rate'] ?? 0);
    
    $discount_amount = $subtotal * ($discount_rate / 100);
    $after_discount = $subtotal - $discount_amount;
    $tax_amount = $after_discount * ($tax_rate / 100);
    $total_amount = $after_discount + $tax_amount;
    
    // Session'a kaydet
    $_SESSION['selected_items'][$quote_token] = $selected_items;
    $_SESSION['cart'][$quote_token] = [
        'quote_id' => $quote['id'],
        'quote_token' => $quote_token,
        'customer_id' => $quote['customer_id'],
        'vehicle_id' => $quote['vehicle_id'],
        'items' => $cart_items,
        'subtotal' => $subtotal,
        'discount_rate' => $discount_rate,
        'discount_amount' => $discount_amount,
        'tax_rate' => $tax_rate,
        'tax_amount' => $tax_amount,
        'total_amount' => $total_amount,
        'created_at' => date('Y-m-d H:i:s')
    ];
    
    echo json_encode([
        'success' => true,
        'message' => 'Sepet güncellendi.',
        'cart_total' => $total_amount,
        'item_count' => count($cart_items),
        'subtotal' => $subtotal,
        'discount_amount' => $discount_amount,
        'tax_amount' => $tax_amount
    ]);
    
} catch (Exception $e) {
    error_log("Update cart error: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
?>