<?php
session_start();

// Debug loglama (çıktı vermeden)
error_log("Process quote debug: Current directory: " . __DIR__);
error_log("Process quote debug: Config file exists: " . (file_exists('config/config.php') ? 'YES' : 'NO'));
error_log("Process quote debug: Functions file exists: " . (file_exists('includes/functions.php') ? 'YES' : 'NO'));

try {
    require_once 'config/config.php';
    error_log("Process quote debug: Config loaded successfully");
} catch (Exception $e) {
    error_log("Config loading error: " . $e->getMessage());
    die("Sistem hatası oluştu. Lütfen daha sonra tekrar deneyin.");
}

try {
    require_once 'includes/functions.php';
    require_once 'includes/appointment_functions.php';
    error_log("Process quote debug: Functions loaded successfully");
} catch (Exception $e) {
    error_log("Functions loading error: " . $e->getMessage());
    die("Sistem hatası oluştu. Lütfen daha sonra tekrar deneyin.");
}

// Debug: PDO bağlantısını kontrol et
error_log("Process quote debug: PDO connection: " . (isset($pdo) ? 'EXISTS' : 'NOT EXISTS'));

// Debug: POST verilerini logla
error_log("Process Quote Debug - POST Data: " . print_r($_POST, true));

// Form işleme
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // Form verilerini al ve doğrula
        $brand = trim($_POST['brand'] ?? '');
        $model = trim($_POST['model'] ?? '');
        $year = intval($_POST['year'] ?? 0);
        $fuel_type = trim($_POST['fuel_type'] ?? '');
        $transmission = trim($_POST['transmission'] ?? '');
        $mileage = intval($_POST['mileage'] ?? 0);
        $plate = trim($_POST['plate'] ?? '');
        $chassis_number = trim($_POST['chassis_number'] ?? '');
        $first_name = trim($_POST['first_name'] ?? '');
        $last_name = trim($_POST['last_name'] ?? '');
        $phone = trim($_POST['phone'] ?? '');
        $email = trim($_POST['email'] ?? '');
        $address = trim($_POST['address'] ?? '');
        $service_type = trim($_POST['service_type'] ?? '');
        $problem_description = trim($_POST['problem_description'] ?? '');
        $preferred_date = trim($_POST['preferred_date'] ?? '');
        $preferred_time = trim($_POST['preferred_time'] ?? '');
        $additional_notes = trim($_POST['additional_notes'] ?? '');
        $kvkk_consent = isset($_POST['kvkk_consent']);
        
        // Validasyon
        $errors = [];
        
        if (empty($brand)) $errors[] = 'Marka seçimi zorunludur.';
        if (empty($model)) $errors[] = 'Model bilgisi zorunludur.';
        if ($year < 1990 || $year > date('Y')) $errors[] = 'Geçerli bir model yılı seçiniz.';
        if (empty($first_name)) $errors[] = 'Ad bilgisi zorunludur.';
        if (empty($last_name)) $errors[] = 'Soyad bilgisi zorunludur.';
        if (empty($phone)) $errors[] = 'Telefon numarası zorunludur.';
        if (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) $errors[] = 'Geçerli bir e-posta adresi giriniz.';
        if (empty($service_type)) $errors[] = 'Hizmet türü seçimi zorunludur.';
        if (!$kvkk_consent) $errors[] = 'KVKK aydınlatma metnini kabul etmelisiniz.';
        
        if (!empty($errors)) {
            $_SESSION['form_errors'] = $errors;
            $_SESSION['form_data'] = $_POST;
            header('Location: quote_form.php');
            exit;
        }
        
        error_log("Process quote debug: Starting database transaction");
        $pdo->beginTransaction();
        
        // Müşteri kontrol et veya oluştur
        error_log("Process quote debug: Checking customer with phone: $phone");
        $stmt = $pdo->prepare("SELECT id FROM customers WHERE phone = ?");
        $stmt->execute([$phone]);
        $customer = $stmt->fetch();
        error_log("Process quote debug: Customer found: " . ($customer ? 'YES (ID: ' . $customer['id'] . ')' : 'NO'));
        
        if ($customer) {
            $customer_id = $customer['id'];
            
            error_log("Process quote debug: Updating existing customer ID: $customer_id");
            // Müşteri bilgilerini güncelle
            $stmt = $pdo->prepare("
                UPDATE customers 
                SET first_name = ?, last_name = ?, email = ?, contact_address = ?, kvkk_consent = ?, kvkk_consent_date = NOW(), updated_at = NOW()
                WHERE id = ?
            ");
            $result = $stmt->execute([$first_name, $last_name, $email, $address, 1, $customer_id]);
            error_log("Process quote debug: Customer update result: " . ($result ? 'SUCCESS' : 'FAILED'));
        } else {
            error_log("Process quote debug: Creating new customer");
            // Yeni müşteri oluştur
            $stmt = $pdo->prepare("
                INSERT INTO customers (first_name, last_name, phone, email, contact_address, kvkk_consent, kvkk_consent_date) 
                VALUES (?, ?, ?, ?, ?, ?, NOW())
            ");
            $result = $stmt->execute([$first_name, $last_name, $phone, $email, $address, 1]);
            $customer_id = $pdo->lastInsertId();
            error_log("Process quote debug: Customer insert result: " . ($result ? 'SUCCESS' : 'FAILED') . ", ID: $customer_id");
        }
        
        // Araç kontrol et veya oluştur
        error_log("Process quote debug: Checking vehicle for customer $customer_id: $brand $model $year");
        $stmt = $pdo->prepare("
            SELECT id FROM vehicles 
            WHERE customer_id = ? AND brand = ? AND model = ? AND year = ?
        ");
        $stmt->execute([$customer_id, $brand, $model, $year]);
        $vehicle = $stmt->fetch();
        error_log("Process quote debug: Vehicle found: " . ($vehicle ? 'YES (ID: ' . $vehicle['id'] . ')' : 'NO'));
        
        if ($vehicle) {
            $vehicle_id = $vehicle['id'];
            
            // Araç bilgilerini güncelle
            $stmt = $pdo->prepare("
                UPDATE vehicles 
                SET plate = ?, fuel_type = ?, transmission = ?, mileage = ?, chassis_number = ?, updated_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$plate ?: null, $fuel_type ?: null, $transmission ?: null, $mileage ?: null, $chassis_number ?: null, $vehicle_id]);
        } else {
            // Yeni araç oluştur
            $stmt = $pdo->prepare("
                INSERT INTO vehicles (customer_id, brand, model, year, fuel_type, transmission, plate, mileage, chassis_number) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([$customer_id, $brand, $model, $year, $fuel_type ?: null, $transmission ?: null, $plate ?: null, $mileage ?: null, $chassis_number ?: null]);
            $vehicle_id = $pdo->lastInsertId();
        }
        
        // Teklif başvurusu oluştur
        error_log("Process quote debug: Creating quote request for customer $customer_id, vehicle $vehicle_id");
        $stmt = $pdo->prepare("
            INSERT INTO quote_requests (customer_id, vehicle_id, service_type, problem_description, preferred_date, preferred_time, additional_notes, status) 
            VALUES (?, ?, ?, ?, ?, ?, ?, 'NEW')
        ");
        $result = $stmt->execute([$customer_id, $vehicle_id, $service_type, $problem_description, $preferred_date ?: null, $preferred_time, $additional_notes]);
        $quote_request_id = $pdo->lastInsertId();
        error_log("Process quote debug: Quote request result: " . ($result ? 'SUCCESS' : 'FAILED') . ", ID: $quote_request_id");
        
        error_log("Process quote debug: Committing transaction");
        $pdo->commit();
        error_log("Process quote debug: Transaction committed successfully");
        
        // Randevu oluşturma (eğer isteniyorsa)
        $appointmentId = null;
        if (!empty($_POST['want_appointment']) && !empty($_POST['appointment_date']) && !empty($_POST['appointment_time'])) {
            try {
                $appointmentData = [
                    'quote_request_id' => $quote_request_id,
                    'customer_id' => $customer_id,
                    'vehicle_id' => $vehicle_id,
                    'appointment_date' => $_POST['appointment_date'],
                    'appointment_time' => $_POST['appointment_time'],
                    'service_type' => $service_type,
                    'notes' => $_POST['appointment_notes'] ?? null
                ];
                
                $appointmentId = createAppointment($appointmentData);
                error_log("Process quote debug: Appointment created with ID: " . $appointmentId);
                
            } catch (Exception $e) {
                error_log("Appointment creation error: " . $e->getMessage());
                // Randevu oluşturulamazsa devam et, ana işlemi etkilemesin
            }
        }
        
        // Başarılı işlem sonrası bilgileri session'a kaydet
        $_SESSION['quote_success'] = [
            'quote_request_id' => $quote_request_id,
            'customer' => [
                'first_name' => $first_name,
                'last_name' => $last_name,
                'phone' => $phone,
                'email' => $email
            ],
            'vehicle' => [
                'brand' => $brand,
                'model' => $model,
                'year' => $year,
                'plate' => $plate
            ],
            'service_type' => $service_type,
            'created_at' => date('Y-m-d H:i:s'),
            'appointment_id' => $appointmentId,
            'appointment_requested' => !empty($_POST['want_appointment'])
        ];
        
        // Thank you sayfasına yönlendir
        header('Location: thank_you.php');
        exit;
        
    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        
        // Detaylı hata loglama
        error_log("Quote process error: " . $e->getMessage());
        error_log("Error file: " . $e->getFile());
        error_log("Error line: " . $e->getLine());
        error_log("Stack trace: " . $e->getTraceAsString());
        
        // Hata durumunda kullanıcıyı bilgilendir
        
        $_SESSION['form_errors'] = ['Bir hata oluştu. Lütfen tekrar deneyiniz. Hata: ' . $e->getMessage()];
        header('Location: quote_form.php');
        exit;
    }
}

// Eğer POST değilse ana sayfaya yönlendir
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: index.php');
    exit;
}
?>