<?php
/**
 * Genel yardımcı fonksiyonlar
 */

/**
 * Güvenli string çıktısı
 */
function h($string) {
    if ($string === null) {
        return '';
    }
    return htmlspecialchars((string)$string, ENT_QUOTES, 'UTF-8');
}

/**
 * CSRF token oluştur
 */
function generateCSRFToken() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

/**
 * CSRF token doğrula
 */
function verifyCSRFToken($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * Rastgele token oluştur
 */
function generateToken($length = 32) {
    return bin2hex(random_bytes($length));
}

/**
 * Telefon numarası formatla
 */
function formatPhone($phone) {
    $phone = preg_replace('/[^0-9]/', '', $phone);
    if (strlen($phone) == 10) {
        $phone = '90' . $phone;
    } elseif (strlen($phone) == 11 && substr($phone, 0, 1) == '0') {
        $phone = '90' . substr($phone, 1);
    }
    return $phone;
}

/**
 * E-posta doğrula
 */
function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * Türkçe karakterleri URL-safe hale getir
 */
function slugify($text) {
    $tr = array('ş','Ş','ı','I','İ','ğ','Ğ','ü','Ü','ö','Ö','Ç','ç');
    $en = array('s','S','i','I','I','g','G','u','U','o','O','C','c');
    $text = str_replace($tr, $en, $text);
    $text = preg_replace('/[^a-zA-Z0-9\-]/', '-', $text);
    $text = preg_replace('/-+/', '-', $text);
    return trim($text, '-');
}

/**
 * Para formatı
 */
function formatMoney($amount, $currency = 'TRY') {
    $symbols = [
        'TRY' => '₺',
        'USD' => '$',
        'EUR' => '€'
    ];
    
    $symbol = $symbols[$currency] ?? $currency;
    return number_format($amount, 2, ',', '.') . ' ' . $symbol;
}

/**
 * Tarih formatı
 */
function formatDate($date, $format = 'd.m.Y H:i') {
    if ($date === null || $date === '') {
        return '-';
    }
    
    try {
        if (is_string($date)) {
            $date = new DateTime($date);
        }
        return $date->format($format);
    } catch (Exception $e) {
        error_log("Date formatting error: " . $e->getMessage());
        return '-';
    }
}

/**
 * Teklif numarası oluştur
 */
function generateQuoteNumber() {
    return 'T' . date('Y') . str_pad(mt_rand(1, 99999), 5, '0', STR_PAD_LEFT);
}

/**
 * Sipariş numarası oluştur
 */
function generateOrderNumber() {
    return 'S' . date('Y') . str_pad(mt_rand(1, 99999), 5, '0', STR_PAD_LEFT);
}

/**
 * SMS gönder (örnek implementasyon)
 */
function sendSMS($phone, $message) {
    // Gerçek SMS API entegrasyonu burada yapılacak
    // Şimdilik log'a yazalım
    error_log("SMS to {$phone}: {$message}");
    return true;
}

/**
 * E-posta gönder - DEPRECATED
 * Bu fonksiyon kaldırıldı. includes/email_functions.php dosyasındaki sendEmail() fonksiyonunu kullanın.
 */
// function sendEmail() fonksiyonu email_functions.php dosyasına taşındı

/**
 * Bildirim gönder
 */
function sendNotification($type, $recipient, $subject, $message, $templateData = null) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            INSERT INTO notifications (type, recipient, subject, message, template_data, status) 
            VALUES (?, ?, ?, ?, ?, 'pending')
        ");
        
        $stmt->execute([
            $type,
            $recipient,
            $subject,
            $message,
            $templateData ? json_encode($templateData) : null
        ]);
        
        $notificationId = $pdo->lastInsertId();
        
        // Hemen göndermeyi dene
        if ($type === 'sms') {
            $success = sendSMS($recipient, $message);
        } elseif ($type === 'email') {
            // E-posta fonksiyonlarını dahil et
            require_once __DIR__ . '/email_functions.php';
            $success = sendEmail($recipient, $subject, $message);
        } else {
            $success = false;
        }
        
        // Durumu güncelle
        $status = $success ? 'sent' : 'failed';
        $stmt = $pdo->prepare("UPDATE notifications SET status = ?, sent_at = NOW() WHERE id = ?");
        $stmt->execute([$status, $notificationId]);
        
        return $success;
        
    } catch (Exception $e) {
        error_log("Notification error: " . $e->getMessage());
        return false;
    }
}

/**
 * Kullanıcı oturum kontrolü
 */
function isLoggedIn() {
    return isset($_SESSION['user_id']);
}

/**
 * Admin kontrolü
 */
function isAdmin() {
    return isset($_SESSION['user_role']) && $_SESSION['user_role'] === 'admin';
}

/**
 * Yetki kontrolü
 */
function hasPermission($permission) {
    if (!isLoggedIn()) return false;
    
    $role = $_SESSION['user_role'] ?? '';
    
    $permissions = [
        'admin' => ['*'], // Tüm yetkiler
        'manager' => ['view_quotes', 'create_quotes', 'edit_quotes', 'view_orders', 'view_reports'],
        'advisor' => ['view_quotes', 'create_quotes', 'edit_quotes', 'view_orders']
    ];
    
    return in_array('*', $permissions[$role] ?? []) || in_array($permission, $permissions[$role] ?? []);
}

/**
 * Yetki kontrolü ile yönlendirme
 */
function requirePermission($permission) {
    if (!hasPermission($permission)) {
        header('Location: /login.php');
        exit;
    }
}

/**
 * Audit log kaydet
 */
function logAudit($action, $tableName = null, $recordId = null, $oldValues = null, $newValues = null) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            INSERT INTO audit_logs (user_id, action, table_name, record_id, old_values, new_values, ip_address, user_agent) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            $_SESSION['user_id'] ?? null,
            $action,
            $tableName,
            $recordId,
            $oldValues ? json_encode($oldValues) : null,
            $newValues ? json_encode($newValues) : null,
            $_SERVER['REMOTE_ADDR'] ?? null,
            $_SERVER['HTTP_USER_AGENT'] ?? null
        ]);
        
    } catch (Exception $e) {
        error_log("Audit log error: " . $e->getMessage());
    }
}

/**
 * Flash mesaj ekle
 */
function addFlashMessage($type, $message) {
    if (!isset($_SESSION['flash_messages'])) {
        $_SESSION['flash_messages'] = [];
    }
    $_SESSION['flash_messages'][] = ['type' => $type, 'message' => $message];
}

/**
 * Flash mesajları al ve temizle
 */
function getFlashMessages() {
    $messages = $_SESSION['flash_messages'] ?? [];
    unset($_SESSION['flash_messages']);
    return $messages;
}

/**
 * Sayfalama hesapla
 */
function calculatePagination($totalRecords, $currentPage, $recordsPerPage = 20) {
    $totalPages = ceil($totalRecords / $recordsPerPage);
    $currentPage = max(1, min($currentPage, $totalPages));
    $offset = ($currentPage - 1) * $recordsPerPage;
    
    return [
        'total_records' => $totalRecords,
        'total_pages' => $totalPages,
        'current_page' => $currentPage,
        'records_per_page' => $recordsPerPage,
        'offset' => $offset,
        'has_previous' => $currentPage > 1,
        'has_next' => $currentPage < $totalPages
    ];
}

/**
 * Dosya yükleme
 */
function uploadFile($file, $directory = 'uploads') {
    if (!isset($file['error']) || $file['error'] !== UPLOAD_ERR_OK) {
        return false;
    }
    
    if ($file['size'] > UPLOAD_MAX_SIZE) {
        return false;
    }
    
    $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    if (!in_array($extension, ALLOWED_FILE_TYPES)) {
        return false;
    }
    
    $filename = uniqid() . '.' . $extension;
    $filepath = $directory . '/' . $filename;
    
    if (!is_dir($directory)) {
        mkdir($directory, 0755, true);
    }
    
    if (move_uploaded_file($file['tmp_name'], $filepath)) {
        return $filename;
    }
    
    return false;
}

/**
 * Şifre hash'le
 */
function hashPassword($password) {
    return password_hash($password, PASSWORD_DEFAULT);
}

/**
 * Şifre doğrula
 */
function verifyPassword($password, $hash) {
    return password_verify($password, $hash);
}

/**
 * Güvenli redirect
 */
function redirect($url) {
    header('Location: ' . $url);
    exit;
}

/**
 * JSON response
 */
function jsonResponse($data, $statusCode = 200) {
    http_response_code($statusCode);
    header('Content-Type: application/json');
    echo json_encode($data);
    exit;
}
?>
