<?php
// E-posta gönderme fonksiyonları

/**
 * PHPMailer ile SMTP e-posta gönder (Önerilen)
 */
function sendSMTPEmail($to, $subject, $message, $attachments = []) {
    // PHPMailer sınıfını kontrol et ve gerekirse yükle
    if (!class_exists('PHPMailer\PHPMailer\PHPMailer')) {
        $phpmailer_loader = __DIR__ . '/../phpmailer_loader.php';
        if (file_exists($phpmailer_loader)) {
            require_once $phpmailer_loader;
        }
        
        if (!class_exists('PHPMailer\PHPMailer\PHPMailer')) {
            error_log("PHPMailer class not found, falling back to simple mail");
            return sendSimpleEmail($to, $subject, $message, $attachments);
        }
    }
    
    try {
        $mail = new PHPMailer\PHPMailer\PHPMailer(true);
        
        // SMTP ayarları
        $mail->isSMTP();
        $mail->Host = SMTP_HOST;
        $mail->SMTPAuth = true;
        $mail->Username = SMTP_USERNAME;
        $mail->Password = SMTP_PASSWORD;
        $mail->SMTPSecure = PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_STARTTLS;
        $mail->Port = SMTP_PORT;
        $mail->CharSet = 'UTF-8';
        
        // Gönderen bilgileri
        $mail->setFrom(SMTP_USERNAME, SITE_NAME);
        $mail->addReplyTo(ADMIN_EMAIL, SITE_NAME);
        
        // Alıcı
        $mail->addAddress($to);
        
        // İçerik
        $mail->isHTML(true);
        $mail->Subject = $subject;
        $mail->Body = $message;
        
        // Ek dosyalar
        if (!empty($attachments)) {
            foreach ($attachments as $attachment) {
                if (is_array($attachment)) {
                    $mail->addAttachment($attachment['path'], $attachment['name'] ?? '');
                } else {
                    $mail->addAttachment($attachment);
                }
            }
        }
        
        $mail->send();
        error_log("SMTP Email sent successfully to: $to");
        return true;
        
    } catch (Exception $e) {
        error_log("SMTP Email failed to: $to - Error: " . $e->getMessage());
        // SMTP başarısız olursa basit mail() fonksiyonunu dene
        return sendSimpleEmail($to, $subject, $message, $attachments);
    }
}

/**
 * Basit PHP mail() fonksiyonu ile e-posta gönder (Yedek)
 */
function sendSimpleEmail($to, $subject, $message, $attachments = []) {
    // mail() fonksiyonunun varlığını kontrol et
    if (!function_exists('mail')) {
        error_log("PHP mail() function is not available on this server");
        return false;
    }
    
    // E-posta adresini kontrol et
    if (empty($to) || !filter_var($to, FILTER_VALIDATE_EMAIL)) {
        error_log("Invalid email address: " . $to);
        return false;
    }
    
    // E-posta başlıkları
    $headers = [];
    $headers[] = 'MIME-Version: 1.0';
    $headers[] = 'Content-type: text/html; charset=UTF-8';
    $headers[] = 'From: ' . SITE_NAME . ' <' . ADMIN_EMAIL . '>';
    $headers[] = 'Reply-To: ' . ADMIN_EMAIL;
    $headers[] = 'X-Mailer: PHP/' . phpversion();
    
    // Ek dosya varsa multipart olarak gönder
    if (!empty($attachments)) {
        return sendEmailWithAttachment($to, $subject, $message, $attachments);
    }
    
    // Basit e-posta gönder
    $success = mail($to, $subject, $message, implode("\r\n", $headers));
    
    if ($success) {
        error_log("Email sent successfully to: $to");
        return true;
    } else {
        error_log("Failed to send email to: $to");
        return false;
    }
}

/**
 * Ek dosyalı e-posta gönder
 */
function sendEmailWithAttachment($to, $subject, $message, $attachments = []) {
    $boundary = md5(time());
    
    // Başlıklar
    $headers = [];
    $headers[] = 'MIME-Version: 1.0';
    $headers[] = 'From: ' . SITE_NAME . ' <' . ADMIN_EMAIL . '>';
    $headers[] = 'Reply-To: ' . ADMIN_EMAIL;
    $headers[] = 'Content-Type: multipart/mixed; boundary="' . $boundary . '"';
    
    // E-posta içeriği
    $body = "--$boundary\r\n";
    $body .= "Content-Type: text/html; charset=UTF-8\r\n";
    $body .= "Content-Transfer-Encoding: 7bit\r\n\r\n";
    $body .= $message . "\r\n";
    
    // Ek dosyalar
    foreach ($attachments as $attachment) {
        if (file_exists($attachment)) {
            $filename = basename($attachment);
            $content = chunk_split(base64_encode(file_get_contents($attachment)));
            
            $body .= "--$boundary\r\n";
            $body .= "Content-Type: application/octet-stream; name=\"$filename\"\r\n";
            $body .= "Content-Disposition: attachment; filename=\"$filename\"\r\n";
            $body .= "Content-Transfer-Encoding: base64\r\n\r\n";
            $body .= $content . "\r\n";
        }
    }
    
    $body .= "--$boundary--";
    
    $success = mail($to, $subject, $body, implode("\r\n", $headers));
    
    if ($success) {
        error_log("Email with attachment sent successfully to: $to");
        return true;
    } else {
        error_log("Failed to send email with attachment to: $to");
        return false;
    }
}

/**
 * Teklif e-postası gönder
 */
function sendQuoteEmail($customer, $quote, $quote_link, $pdf_filename = null) {
    // Müşteri bilgilerini kontrol et
    if (!$customer || !is_array($customer) || empty($customer['email'])) {
        error_log("Invalid customer data for email: " . print_r($customer, true));
        return false;
    }
    
    $subject = "Aracınız İçin Teklif Hazır - " . SITE_NAME;
    
    $message = generateQuoteEmailHTML($customer, $quote, $quote_link);
    
    $attachments = [];
    if ($pdf_filename && file_exists("uploads/quotes/$pdf_filename")) {
        $attachments[] = "uploads/quotes/$pdf_filename";
    }
    
    return sendEmail($customer['email'], $subject, $message, $attachments);
}

/**
 * Teklif e-posta HTML şablonu
 */
function generateQuoteEmailHTML($customer, $quote, $quote_link) {
    // Müşteri bilgilerini kontrol et
    if (!$customer || !is_array($customer)) {
        return '<p>Müşteri bilgileri bulunamadı.</p>';
    }
    
    $customerName = ($customer['first_name'] ?? '') . ' ' . ($customer['last_name'] ?? '');
    
    $html = '
    <!DOCTYPE html>
    <html lang="tr">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Teklif - ' . SITE_NAME . '</title>
        <style>
            body {
                font-family: Arial, sans-serif;
                line-height: 1.6;
                color: #333;
                max-width: 600px;
                margin: 0 auto;
                padding: 20px;
                background-color: #f4f4f4;
            }
            .container {
                background: white;
                padding: 30px;
                border-radius: 10px;
                box-shadow: 0 0 20px rgba(0,0,0,0.1);
            }
            .header {
                text-align: center;
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: white;
                padding: 30px;
                border-radius: 10px;
                margin-bottom: 30px;
            }
            .header h1 {
                margin: 0;
                font-size: 28px;
            }
            .content {
                padding: 0 20px;
            }
            .quote-info {
                background: #f8f9fa;
                padding: 20px;
                border-radius: 8px;
                margin: 20px 0;
                border-left: 4px solid #667eea;
            }
            .quote-info h3 {
                margin-top: 0;
                color: #667eea;
            }
            .cta-button {
                display: inline-block;
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: white;
                padding: 15px 30px;
                text-decoration: none;
                border-radius: 8px;
                font-weight: bold;
                margin: 20px 0;
                text-align: center;
            }
            .footer {
                text-align: center;
                margin-top: 30px;
                padding-top: 20px;
                border-top: 1px solid #eee;
                color: #666;
                font-size: 14px;
            }
            .contact-info {
                background: #e3f2fd;
                padding: 15px;
                border-radius: 8px;
                margin: 20px 0;
            }
        </style>
    </head>
    <body>
        <div class="container">
            <div class="header">
                <h1>' . SITE_NAME . '</h1>
                <p>Profesyonel Araç Bakım ve Onarım Hizmetleri</p>
            </div>
            
            <div class="content">
                <h2>Sayın ' . htmlspecialchars($customerName) . ',</h2>
                
                <p>Aracınız için talep ettiğiniz <strong>ücretsiz teklif</strong> hazırlandı!</p>
                
                <div class="quote-info">
                    <h3>📋 Teklif Bilgileri</h3>
                    <p><strong>Teklif No:</strong> ' . htmlspecialchars($quote['quote_number']) . '</p>
                    <p><strong>Tarih:</strong> ' . date('d.m.Y H:i', strtotime($quote['created_at'])) . '</p>
                    <p><strong>Geçerlilik:</strong> ' . date('d.m.Y H:i', strtotime($quote['valid_until'])) . '</p>
                    <p><strong>Toplam Tutar:</strong> <span style="color: #28a745; font-size: 18px; font-weight: bold;">' . number_format($quote['total_amount'], 2) . ' ₺</span></p>
                </div>
                
                <p>Teklifinizi görüntülemek, ürün/hizmetleri seçmek ve online ödeme yapmak için aşağıdaki butona tıklayınız:</p>
                
                <div style="text-align: center;">
                    <a href="' . htmlspecialchars($quote_link) . '" class="cta-button">
                        🚗 TEKLİFİ GÖRÜNTÜLE VE ONAYLA
                    </a>
                </div>
                
                <div class="contact-info">
                    <h4>📞 İletişim Bilgileri</h4>
                    <p><strong>Telefon:</strong> 0212 123 45 67</p>
                    <p><strong>WhatsApp:</strong> 0532 123 45 67</p>
                    <p><strong>E-posta:</strong> ' . ADMIN_EMAIL . '</p>
                    <p><strong>Adres:</strong> Örnek Mahalle, Araç Bakım Caddesi No:123, İstanbul</p>
                </div>
                
                <h3>✨ Neden Bizi Tercih Etmelisiniz?</h3>
                <ul>
                    <li>✅ <strong>15 yıllık</strong> tecrübe</li>
                    <li>✅ <strong>Sertifikalı</strong> uzman kadro</li>
                    <li>✅ <strong>1 yıl</strong> garanti</li>
                    <li>✅ <strong>Ücretsiz</strong> ön inceleme</li>
                    <li>✅ <strong>Orijinal</strong> yedek parça</li>
                    <li>✅ <strong>Hızlı</strong> ve güvenilir hizmet</li>
                </ul>
                
                <p><strong>Not:</strong> Bu teklif ' . date('d.m.Y H:i', strtotime($quote['valid_until'])) . ' tarihine kadar geçerlidir. Sorularınız için bizi arayabilirsiniz.</p>
            </div>
            
            <div class="footer">
                <p>Bu e-posta otomatik olarak gönderilmiştir.</p>
                <p>' . SITE_NAME . ' | Profesyonel Araç Bakım Hizmetleri</p>
                <p>© ' . date('Y') . ' Tüm hakları saklıdır.</p>
            </div>
        </div>
    </body>
    </html>';
    
    return $html;
}

/**
 * Test e-postası gönder
 */
function sendTestEmail($to = null) {
    $to = $to ?: ADMIN_EMAIL;
    $subject = "Test E-posta - " . SITE_NAME;
    $message = "
    <h2>Test E-posta</h2>
    <p>Bu bir test e-postasıdır. E-posta sistemi düzgün çalışıyor.</p>
    <p><strong>Gönderim Zamanı:</strong> " . date('d.m.Y H:i:s') . "</p>
    <p><strong>Sunucu:</strong> " . $_SERVER['SERVER_NAME'] . "</p>
    ";
    
    return sendEmail($to, $subject, $message);
}

/**
 * Sipariş onay e-postası gönder
 */
function sendOrderConfirmationEmail($customer, $order) {
    $subject = "Siparişiniz Onaylandı - " . SITE_NAME;
    
    $customerName = $customer['first_name'] . ' ' . $customer['last_name'];
    
    $message = '
    <h2>Siparişiniz Onaylandı!</h2>
    <p>Sayın ' . htmlspecialchars($customerName) . ',</p>
    <p>Siparişiniz başarıyla alındı ve işleme konuldu.</p>
    <p><strong>Sipariş No:</strong> ' . htmlspecialchars($order['order_number']) . '</p>
    <p><strong>Toplam Tutar:</strong> ' . number_format($order['total_amount'], 2) . ' ₺</p>
    <p>En kısa sürede sizinle iletişime geçeceğiz.</p>
    ';
    
    return sendEmail($customer['email'], $subject, $message);
}

/**
 * Ana e-posta gönderme fonksiyonu
 * Önce SMTP, başarısız olursa basit mail() kullanır
 */
if (!function_exists('sendEmail')) {
function sendEmail($to, $subject, $message, $attachments = []) {
    // E-posta adresini kontrol et
    if (empty($to) || !filter_var($to, FILTER_VALIDATE_EMAIL)) {
        error_log("Invalid email address for sendEmail: " . $to);
        return false;
    }
    
    // Önce SMTP ile dene
    if (defined('SMTP_HOST') && defined('SMTP_USERNAME') && defined('SMTP_PASSWORD') && 
        SMTP_HOST !== 'smtp.gmail.com' && SMTP_USERNAME !== 'your_email@gmail.com') {
        $result = sendSMTPEmail($to, $subject, $message, $attachments);
        if ($result) {
            return true;
        }
        error_log("SMTP failed, trying fallback mail() function");
    } else {
        error_log("SMTP not configured, using fallback mail() function");
    }
    
    // SMTP başarısız olursa basit mail() kullan
    return sendSimpleEmail($to, $subject, $message, $attachments);
}
} // function_exists('sendEmail') kapanışı
?>
