<?php
/**
 * Randevu sistemi fonksiyonları
 */

/**
 * Çalışma saatlerini getir
 */
function getWorkingHours() {
    global $pdo;
    
    $stmt = $pdo->prepare("
        SELECT day_of_week, start_time, end_time, break_start, break_end, is_active 
        FROM working_hours 
        ORDER BY day_of_week
    ");
    $stmt->execute();
    
    $hours = [];
    while ($row = $stmt->fetch()) {
        $hours[$row['day_of_week']] = $row;
    }
    
    return $hours;
}

/**
 * Belirli bir tarih için çalışma saatlerini kontrol et
 */
function isWorkingDay($date) {
    $dayOfWeek = date('w', strtotime($date)); // 0=Pazar, 1=Pazartesi, ...
    $workingHours = getWorkingHours();
    
    // Çalışma günü mü kontrol et
    if (!isset($workingHours[$dayOfWeek]) || !$workingHours[$dayOfWeek]['is_active']) {
        return false;
    }
    
    // Tatil günü mü kontrol et
    if (isHoliday($date)) {
        return false;
    }
    
    return true;
}

/**
 * Tatil günü kontrolü
 */
function isHoliday($date) {
    global $pdo;
    
    $stmt = $pdo->prepare("
        SELECT COUNT(*) FROM holidays 
        WHERE holiday_date = ? AND is_active = 1
    ");
    $stmt->execute([$date]);
    
    return $stmt->fetchColumn() > 0;
}

/**
 * Belirli bir tarih için müsait saatleri getir
 */
function getAvailableTimeSlots($date) {
    global $pdo;
    
    // Çalışma günü değilse boş array döndür
    if (!isWorkingDay($date)) {
        return [];
    }
    
    $dayOfWeek = date('w', strtotime($date));
    $workingHours = getWorkingHours();
    $dayHours = $workingHours[$dayOfWeek];
    
    // Randevu ayarlarını al
    $slotDuration = getAppointmentSetting('slot_duration', 60);
    
    // Çalışma saatleri içinde slot'ları oluştur
    $slots = [];
    $startTime = strtotime($dayHours['start_time']);
    $endTime = strtotime($dayHours['end_time']);
    $breakStart = $dayHours['break_start'] ? strtotime($dayHours['break_start']) : null;
    $breakEnd = $dayHours['break_end'] ? strtotime($dayHours['break_end']) : null;
    
    $currentTime = $startTime;
    while ($currentTime < $endTime) {
        $slotTime = date('H:i:s', $currentTime);
        
        // Mola saatlerini atla
        if ($breakStart && $breakEnd && $currentTime >= $breakStart && $currentTime < $breakEnd) {
            $currentTime = $breakEnd;
            continue;
        }
        
        // Slot'un bitişi çalışma saatini aşmasın
        if (($currentTime + ($slotDuration * 60)) > $endTime) {
            break;
        }
        
        $slots[] = $slotTime;
        $currentTime += ($slotDuration * 60);
    }
    
    // Mevcut randevuları kontrol et ve dolu saatleri çıkar
    $stmt = $pdo->prepare("
        SELECT appointment_time, duration_minutes 
        FROM appointments 
        WHERE appointment_date = ? 
        AND status NOT IN ('CANCELLED', 'NO_SHOW')
    ");
    $stmt->execute([$date]);
    $bookedSlots = $stmt->fetchAll();
    
    // Dolu saatleri filtrele
    $availableSlots = [];
    foreach ($slots as $slot) {
        $isAvailable = true;
        $slotStart = strtotime($slot);
        
        foreach ($bookedSlots as $booked) {
            $bookedStart = strtotime($booked['appointment_time']);
            $bookedEnd = $bookedStart + ($booked['duration_minutes'] * 60);
            
            // Slot çakışıyor mu kontrol et
            if ($slotStart < $bookedEnd && ($slotStart + ($slotDuration * 60)) > $bookedStart) {
                $isAvailable = false;
                break;
            }
        }
        
        if ($isAvailable) {
            $availableSlots[] = $slot;
        }
    }
    
    return $availableSlots;
}

/**
 * Randevu ayarı getir
 */
function getAppointmentSetting($key, $default = null) {
    global $pdo;
    
    $stmt = $pdo->prepare("SELECT setting_value FROM appointment_settings WHERE setting_key = ?");
    $stmt->execute([$key]);
    $result = $stmt->fetchColumn();
    
    return $result !== false ? $result : $default;
}

/**
 * Randevu oluştur
 */
function createAppointment($data) {
    global $pdo;
    
    try {
        $pdo->beginTransaction();
        
        // Randevu saatinin müsait olduğunu kontrol et
        $availableSlots = getAvailableTimeSlots($data['appointment_date']);
        if (!in_array($data['appointment_time'], $availableSlots)) {
            throw new Exception('Seçilen saat artık müsait değil.');
        }
        
        // Randevuyu oluştur
        $stmt = $pdo->prepare("
            INSERT INTO appointments (
                quote_request_id, customer_id, vehicle_id, appointment_date, 
                appointment_time, duration_minutes, service_type, status, notes
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            $data['quote_request_id'] ?? null,
            $data['customer_id'],
            $data['vehicle_id'],
            $data['appointment_date'],
            $data['appointment_time'],
            $data['duration_minutes'] ?? 60,
            $data['service_type'],
            'PENDING',
            $data['notes'] ?? null
        ]);
        
        $appointmentId = $pdo->lastInsertId();
        
        $pdo->commit();
        return $appointmentId;
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}

/**
 * Randevu durumunu güncelle
 */
function updateAppointmentStatus($appointmentId, $status, $adminNotes = null) {
    global $pdo;
    
    $stmt = $pdo->prepare("
        UPDATE appointments 
        SET status = ?, admin_notes = ?, updated_at = NOW() 
        WHERE id = ?
    ");
    
    return $stmt->execute([$status, $adminNotes, $appointmentId]);
}

/**
 * Müşterinin randevularını getir
 */
function getCustomerAppointments($customerId, $limit = 10) {
    global $pdo;
    
    $stmt = $pdo->prepare("
        SELECT a.*, v.brand, v.model, v.year, v.plate
        FROM appointments a
        JOIN vehicles v ON a.vehicle_id = v.id
        WHERE a.customer_id = ?
        ORDER BY a.appointment_date DESC, a.appointment_time DESC
        LIMIT ?
    ");
    $stmt->execute([$customerId, $limit]);
    
    return $stmt->fetchAll();
}

/**
 * Admin için randevuları getir
 */
function getAppointments($filters = [], $limit = 50, $offset = 0) {
    global $pdo;
    
    $where = ['1=1'];
    $params = [];
    
    if (!empty($filters['date_from'])) {
        $where[] = 'a.appointment_date >= ?';
        $params[] = $filters['date_from'];
    }
    
    if (!empty($filters['date_to'])) {
        $where[] = 'a.appointment_date <= ?';
        $params[] = $filters['date_to'];
    }
    
    if (!empty($filters['status'])) {
        $where[] = 'a.status = ?';
        $params[] = $filters['status'];
    }
    
    if (!empty($filters['search'])) {
        $where[] = '(c.first_name LIKE ? OR c.last_name LIKE ? OR v.plate LIKE ?)';
        $searchTerm = '%' . $filters['search'] . '%';
        $params[] = $searchTerm;
        $params[] = $searchTerm;
        $params[] = $searchTerm;
    }
    
    $whereClause = implode(' AND ', $where);
    
    $stmt = $pdo->prepare("
        SELECT a.*, 
               c.first_name, c.last_name, c.phone, c.email,
               v.brand, v.model, v.year, v.plate,
               u.first_name as tech_first_name, u.last_name as tech_last_name
        FROM appointments a
        JOIN customers c ON a.customer_id = c.id
        JOIN vehicles v ON a.vehicle_id = v.id
        LEFT JOIN users u ON a.assigned_technician_id = u.id
        WHERE $whereClause
        ORDER BY a.appointment_date ASC, a.appointment_time ASC
        LIMIT ? OFFSET ?
    ");
    
    $params[] = $limit;
    $params[] = $offset;
    $stmt->execute($params);
    
    return $stmt->fetchAll();
}

/**
 * Randevu sayısını getir
 */
function getAppointmentsCount($filters = []) {
    global $pdo;
    
    $where = ['1=1'];
    $params = [];
    
    if (!empty($filters['date_from'])) {
        $where[] = 'a.appointment_date >= ?';
        $params[] = $filters['date_from'];
    }
    
    if (!empty($filters['date_to'])) {
        $where[] = 'a.appointment_date <= ?';
        $params[] = $filters['date_to'];
    }
    
    if (!empty($filters['status'])) {
        $where[] = 'a.status = ?';
        $params[] = $filters['status'];
    }
    
    if (!empty($filters['search'])) {
        $where[] = '(c.first_name LIKE ? OR c.last_name LIKE ? OR v.plate LIKE ?)';
        $searchTerm = '%' . $filters['search'] . '%';
        $params[] = $searchTerm;
        $params[] = $searchTerm;
        $params[] = $searchTerm;
    }
    
    $whereClause = implode(' AND ', $where);
    
    $stmt = $pdo->prepare("
        SELECT COUNT(*) 
        FROM appointments a
        JOIN customers c ON a.customer_id = c.id
        JOIN vehicles v ON a.vehicle_id = v.id
        WHERE $whereClause
    ");
    
    $stmt->execute($params);
    return $stmt->fetchColumn();
}

/**
 * Günlük randevu sayısını getir
 */
function getDailyAppointmentCount($date) {
    global $pdo;
    
    $stmt = $pdo->prepare("
        SELECT COUNT(*) FROM appointments 
        WHERE appointment_date = ? 
        AND status NOT IN ('CANCELLED', 'NO_SHOW')
    ");
    $stmt->execute([$date]);
    
    return $stmt->fetchColumn();
}

/**
 * Randevu hatırlatması gönder
 */
function sendAppointmentReminder($appointmentId) {
    global $pdo;
    
    $stmt = $pdo->prepare("
        SELECT a.*, c.first_name, c.last_name, c.email, c.phone,
               v.brand, v.model, v.year
        FROM appointments a
        JOIN customers c ON a.customer_id = c.id
        JOIN vehicles v ON a.vehicle_id = v.id
        WHERE a.id = ?
    ");
    $stmt->execute([$appointmentId]);
    $appointment = $stmt->fetch();
    
    if (!$appointment) {
        return false;
    }
    
    // E-posta hatırlatması gönder
    if (!empty($appointment['email'])) {
        require_once __DIR__ . '/email_functions.php';
        
        $subject = "Randevu Hatırlatması - " . SITE_NAME;
        $appointmentDate = formatDate($appointment['appointment_date'] . ' ' . $appointment['appointment_time'], 'd.m.Y H:i');
        
        $message = "
        <div style='font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;'>
            <h2>🕐 Randevu Hatırlatması</h2>
            <p>Sayın {$appointment['first_name']} {$appointment['last_name']},</p>
            <p>Yarın randevunuz bulunmaktadır:</p>
            
            <div style='background: #f8f9fa; padding: 20px; border-radius: 8px; margin: 20px 0;'>
                <p><strong>📅 Tarih:</strong> $appointmentDate</p>
                <p><strong>🚗 Araç:</strong> {$appointment['brand']} {$appointment['model']} ({$appointment['year']})</p>
                <p><strong>🔧 Hizmet:</strong> {$appointment['service_type']}</p>
            </div>
            
            <p>Randevunuza zamanında gelmenizi rica ederiz.</p>
            <p>İletişim: " . ADMIN_EMAIL . "</p>
        </div>
        ";
        
        $emailSent = sendEmail($appointment['email'], $subject, $message);
        
        if ($emailSent) {
            // Hatırlatma gönderildi olarak işaretle
            $stmt = $pdo->prepare("UPDATE appointments SET reminder_sent = 1 WHERE id = ?");
            $stmt->execute([$appointmentId]);
        }
        
        return $emailSent;
    }
    
    return false;
}

/**
 * Randevu durumu için Türkçe açıklama
 */
function getAppointmentStatusText($status) {
    $statuses = [
        'PENDING' => 'Beklemede',
        'CONFIRMED' => 'Onaylandı',
        'IN_PROGRESS' => 'Devam Ediyor',
        'COMPLETED' => 'Tamamlandı',
        'CANCELLED' => 'İptal Edildi',
        'NO_SHOW' => 'Gelmedi'
    ];
    
    return $statuses[$status] ?? $status;
}

/**
 * Randevu durumu için CSS sınıfı
 */
function getAppointmentStatusClass($status) {
    $classes = [
        'PENDING' => 'warning',
        'CONFIRMED' => 'info',
        'IN_PROGRESS' => 'primary',
        'COMPLETED' => 'success',
        'CANCELLED' => 'danger',
        'NO_SHOW' => 'secondary'
    ];
    
    return $classes[$status] ?? 'secondary';
}
?>
