/**
 * Araç Bakım Servisi - Ana JavaScript Dosyası
 */

// DOM yüklendiğinde çalışacak fonksiyonlar
document.addEventListener('DOMContentLoaded', function() {
    // Form validasyonu
    initFormValidation();
    
    // Telefon formatlaması
    initPhoneFormatting();
    
    // Loading overlay
    initLoadingOverlay();
    
    // Tooltips
    initTooltips();
    
    // Auto-hide alerts
    initAutoHideAlerts();
    
    // Sidebar toggle (mobile)
    initSidebarToggle();
});

/**
 * Form validasyonu
 */
function initFormValidation() {
    const forms = document.querySelectorAll('.needs-validation');
    
    forms.forEach(form => {
        form.addEventListener('submit', function(event) {
            if (!form.checkValidity()) {
                event.preventDefault();
                event.stopPropagation();
            }
            
            form.classList.add('was-validated');
        });
    });
    
    // Teklif formu özel validasyonu
    const quoteForm = document.getElementById('quoteForm');
    if (quoteForm) {
        quoteForm.addEventListener('submit', function(event) {
            event.preventDefault();
            
            if (validateQuoteForm()) {
                showLoading();
                submitQuoteForm();
            }
        });
    }
}

/**
 * Teklif formu validasyonu
 */
function validateQuoteForm() {
    let isValid = true;
    const errors = [];
    
    // Gerekli alanları kontrol et
    const requiredFields = [
        { id: 'brand', name: 'Marka' },
        { id: 'model', name: 'Model' },
        { id: 'year', name: 'Model Yılı' },
        { id: 'fuel_type', name: 'Yakıt Türü' },
        { id: 'transmission', name: 'Şanzıman' },
        { id: 'first_name', name: 'Ad' },
        { id: 'last_name', name: 'Soyad' },
        { id: 'phone', name: 'Telefon' }
    ];
    
    requiredFields.forEach(field => {
        const element = document.getElementById(field.id);
        if (!element.value.trim()) {
            errors.push(`${field.name} alanı zorunludur.`);
            element.classList.add('is-invalid');
            isValid = false;
        } else {
            element.classList.remove('is-invalid');
        }
    });
    
    // Telefon numarası kontrolü
    const phone = document.getElementById('phone').value.trim();
    if (phone && !isValidPhone(phone)) {
        errors.push('Geçerli bir telefon numarası giriniz.');
        document.getElementById('phone').classList.add('is-invalid');
        isValid = false;
    }
    
    // E-posta kontrolü (opsiyonel)
    const email = document.getElementById('email').value.trim();
    if (email && !isValidEmail(email)) {
        errors.push('Geçerli bir e-posta adresi giriniz.');
        document.getElementById('email').classList.add('is-invalid');
        isValid = false;
    }
    
    // KVKK onayı kontrolü
    const kvkkConsent = document.getElementById('kvkk_consent');
    if (!kvkkConsent.checked) {
        errors.push('KVKK aydınlatma metnini kabul etmelisiniz.');
        kvkkConsent.classList.add('is-invalid');
        isValid = false;
    }
    
    // Hataları göster
    if (!isValid) {
        showAlert('danger', errors.join('<br>'));
    }
    
    return isValid;
}

/**
 * Teklif formunu gönder
 */
function submitQuoteForm() {
    const formData = new FormData(document.getElementById('quoteForm'));
    
    fetch('process_quote.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        hideLoading();
        
        if (data.success) {
            showAlert('success', data.message);
            document.getElementById('quoteForm').reset();
            
            // Başarı sayfasına yönlendir
            if (data.redirect) {
                setTimeout(() => {
                    window.location.href = data.redirect;
                }, 2000);
            }
        } else {
            showAlert('danger', data.message);
        }
    })
    .catch(error => {
        hideLoading();
        console.error('Error:', error);
        showAlert('danger', 'Bir hata oluştu. Lütfen tekrar deneyiniz.');
    });
}

/**
 * Telefon formatlaması
 */
function initPhoneFormatting() {
    const phoneInputs = document.querySelectorAll('input[type="tel"]');
    
    phoneInputs.forEach(input => {
        input.addEventListener('input', function(e) {
            let value = e.target.value.replace(/\D/g, '');
            
            if (value.length > 0) {
                if (value.length <= 3) {
                    value = value;
                } else if (value.length <= 6) {
                    value = value.slice(0, 3) + ' ' + value.slice(3);
                } else if (value.length <= 8) {
                    value = value.slice(0, 3) + ' ' + value.slice(3, 6) + ' ' + value.slice(6);
                } else {
                    value = value.slice(0, 3) + ' ' + value.slice(3, 6) + ' ' + value.slice(6, 8) + ' ' + value.slice(8, 10);
                }
            }
            
            e.target.value = value;
        });
    });
}

/**
 * Loading overlay
 */
function initLoadingOverlay() {
    // Loading overlay HTML'i oluştur
    const loadingOverlay = document.createElement('div');
    loadingOverlay.id = 'loadingOverlay';
    loadingOverlay.className = 'loading-overlay';
    loadingOverlay.style.display = 'none';
    loadingOverlay.innerHTML = `
        <div class="text-center">
            <div class="spinner-border text-primary" role="status">
                <span class="visually-hidden">Yükleniyor...</span>
            </div>
            <div class="mt-3 text-white">
                <strong>Lütfen bekleyiniz...</strong>
            </div>
        </div>
    `;
    
    document.body.appendChild(loadingOverlay);
}

function showLoading() {
    document.getElementById('loadingOverlay').style.display = 'flex';
}

function hideLoading() {
    document.getElementById('loadingOverlay').style.display = 'none';
}

/**
 * Alert göster
 */
function showAlert(type, message) {
    const alertContainer = document.getElementById('alertContainer') || createAlertContainer();
    
    const alertDiv = document.createElement('div');
    alertDiv.className = `alert alert-${type} alert-dismissible fade show`;
    alertDiv.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    alertContainer.appendChild(alertDiv);
    
    // 5 saniye sonra otomatik kapat
    setTimeout(() => {
        if (alertDiv.parentNode) {
            alertDiv.remove();
        }
    }, 5000);
}

function createAlertContainer() {
    const container = document.createElement('div');
    container.id = 'alertContainer';
    container.className = 'position-fixed top-0 end-0 p-3';
    container.style.zIndex = '9999';
    document.body.appendChild(container);
    return container;
}

/**
 * Tooltips
 */
function initTooltips() {
    const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    tooltipTriggerList.map(function (tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl);
    });
}

/**
 * Auto-hide alerts
 */
function initAutoHideAlerts() {
    const alerts = document.querySelectorAll('.alert:not(.alert-permanent)');
    alerts.forEach(alert => {
        setTimeout(() => {
            if (alert.parentNode) {
                alert.classList.remove('show');
                setTimeout(() => alert.remove(), 150);
            }
        }, 5000);
    });
}

/**
 * Sidebar toggle (mobile)
 */
function initSidebarToggle() {
    const sidebarToggle = document.getElementById('sidebarToggle');
    const sidebar = document.querySelector('.sidebar');
    
    if (sidebarToggle && sidebar) {
        sidebarToggle.addEventListener('click', function() {
            sidebar.classList.toggle('show');
        });
        
        // Sidebar dışına tıklandığında kapat
        document.addEventListener('click', function(event) {
            if (!sidebar.contains(event.target) && !sidebarToggle.contains(event.target)) {
                sidebar.classList.remove('show');
            }
        });
    }
}

/**
 * Validasyon fonksiyonları
 */
function isValidEmail(email) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
}

function isValidPhone(phone) {
    const phoneRegex = /^(\+90|0)?[5][0-9]{9}$/;
    const cleanPhone = phone.replace(/\s/g, '');
    return phoneRegex.test(cleanPhone);
}

/**
 * Para formatlaması
 */
function formatMoney(amount, currency = 'TRY') {
    const formatter = new Intl.NumberFormat('tr-TR', {
        style: 'currency',
        currency: currency,
        minimumFractionDigits: 2
    });
    return formatter.format(amount);
}

/**
 * Tarih formatlaması
 */
function formatDate(date, options = {}) {
    const defaultOptions = {
        year: 'numeric',
        month: '2-digit',
        day: '2-digit',
        hour: '2-digit',
        minute: '2-digit'
    };
    
    const formatter = new Intl.DateTimeFormat('tr-TR', { ...defaultOptions, ...options });
    return formatter.format(new Date(date));
}

/**
 * Debounce fonksiyonu
 */
function debounce(func, wait, immediate) {
    let timeout;
    return function executedFunction() {
        const context = this;
        const args = arguments;
        const later = function() {
            timeout = null;
            if (!immediate) func.apply(context, args);
        };
        const callNow = immediate && !timeout;
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
        if (callNow) func.apply(context, args);
    };
}

/**
 * AJAX yardımcı fonksiyonları
 */
function ajaxGet(url, callback) {
    fetch(url)
        .then(response => response.json())
        .then(data => callback(null, data))
        .catch(error => callback(error, null));
}

function ajaxPost(url, data, callback) {
    const formData = new FormData();
    
    for (const key in data) {
        formData.append(key, data[key]);
    }
    
    fetch(url, {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => callback(null, data))
    .catch(error => callback(error, null));
}

/**
 * Local Storage yardımcıları
 */
function setLocalStorage(key, value) {
    try {
        localStorage.setItem(key, JSON.stringify(value));
        return true;
    } catch (e) {
        console.error('LocalStorage error:', e);
        return false;
    }
}

function getLocalStorage(key, defaultValue = null) {
    try {
        const item = localStorage.getItem(key);
        return item ? JSON.parse(item) : defaultValue;
    } catch (e) {
        console.error('LocalStorage error:', e);
        return defaultValue;
    }
}

/**
 * Sayfa animasyonları
 */
function animateOnScroll() {
    const elements = document.querySelectorAll('.fade-in');
    
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.style.opacity = '1';
                entry.target.style.transform = 'translateY(0)';
            }
        });
    });
    
    elements.forEach(element => {
        element.style.opacity = '0';
        element.style.transform = 'translateY(20px)';
        element.style.transition = 'opacity 0.6s ease, transform 0.6s ease';
        observer.observe(element);
    });
}

// Sayfa yüklendiğinde animasyonları başlat
window.addEventListener('load', animateOnScroll);

/**
 * Print fonksiyonu
 */
function printPage() {
    window.print();
}

/**
 * Copy to clipboard
 */
function copyToClipboard(text) {
    navigator.clipboard.writeText(text).then(() => {
        showAlert('success', 'Panoya kopyalandı!');
    }).catch(err => {
        console.error('Copy failed:', err);
        showAlert('danger', 'Kopyalama başarısız!');
    });
}

/**
 * Confirm dialog
 */
function confirmAction(message, callback) {
    if (confirm(message)) {
        callback();
    }
}

/**
 * Auto-save form data
 */
function initAutoSave(formId, storageKey) {
    const form = document.getElementById(formId);
    if (!form) return;
    
    // Form verilerini yükle
    const savedData = getLocalStorage(storageKey);
    if (savedData) {
        Object.keys(savedData).forEach(key => {
            const element = form.querySelector(`[name="${key}"]`);
            if (element) {
                element.value = savedData[key];
            }
        });
    }
    
    // Form değişikliklerini kaydet
    const saveFormData = debounce(() => {
        const formData = new FormData(form);
        const data = {};
        for (const [key, value] of formData.entries()) {
            data[key] = value;
        }
        setLocalStorage(storageKey, data);
    }, 1000);
    
    form.addEventListener('input', saveFormData);
    form.addEventListener('change', saveFormData);
    
    // Form gönderildiğinde kayıtlı veriyi temizle
    form.addEventListener('submit', () => {
        localStorage.removeItem(storageKey);
    });
}

// Global error handler
window.addEventListener('error', function(e) {
    console.error('Global error:', e.error);
    // Geliştirme ortamında hataları göster
    if (window.location.hostname === 'localhost') {
        showAlert('danger', 'JavaScript Hatası: ' + e.message);
    }
});

// Unhandled promise rejection handler
window.addEventListener('unhandledrejection', function(e) {
    console.error('Unhandled promise rejection:', e.reason);
    if (window.location.hostname === 'localhost') {
        showAlert('warning', 'Promise Hatası: ' + e.reason);
    }
});

// Export functions for use in other scripts
window.AracBakim = {
    showAlert,
    showLoading,
    hideLoading,
    formatMoney,
    formatDate,
    isValidEmail,
    isValidPhone,
    ajaxGet,
    ajaxPost,
    copyToClipboard,
    confirmAction,
    printPage
};
