<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';

$pageTitle = 'Kullanıcılar';
$currentPage = 'users';

if (!isLoggedIn() || !isAdmin()) {
    redirect('login.php');
}

// Sadece admin kullanıcıları görebilir
if ($_SESSION['user_role'] !== 'admin') {
    addFlashMessage('error', 'Bu sayfaya erişim yetkiniz yok.');
    redirect('dashboard.php');
}

$success_message = '';
$error_message = '';

// Kullanıcıları al
try {
    $stmt = $pdo->query("
        SELECT u.*, 
               COUNT(DISTINCT q.id) as quote_count,
               COUNT(DISTINCT o.id) as order_count,
               MAX(u.last_login) as last_activity
        FROM users u
        LEFT JOIN quotes q ON u.id = q.created_by
        LEFT JOIN orders o ON u.id = o.created_by
        GROUP BY u.id
        ORDER BY u.created_at DESC
    ");
    $users = $stmt->fetchAll();
} catch (Exception $e) {
    $users = [];
    $error_message = 'Kullanıcılar yüklenirken hata oluştu.';
}

// Kullanıcı silme
if (isset($_POST['delete_user'])) {
    $user_id = intval($_POST['user_id']);
    
    if ($user_id === $_SESSION['user_id']) {
        $error_message = 'Kendi hesabınızı silemezsiniz.';
    } else {
        try {
            // Kullanıcının verilerini kontrol et
            $stmt = $pdo->prepare("
                SELECT 
                    (SELECT COUNT(*) FROM quotes WHERE created_by = ?) as quote_count,
                    (SELECT COUNT(*) FROM orders WHERE created_by = ?) as order_count
            ");
            $stmt->execute([$user_id, $user_id]);
            $user_data = $stmt->fetch();
            
            if ($user_data['quote_count'] > 0 || $user_data['order_count'] > 0) {
                $error_message = 'Bu kullanıcının teklif veya siparişleri bulunduğu için silinemez. Önce pasif hale getiriniz.';
            } else {
                $stmt = $pdo->prepare("DELETE FROM users WHERE id = ?");
                $stmt->execute([$user_id]);
                
                logAudit('user_deleted', 'users', $user_id, null, ['deleted_by' => $_SESSION['user_id']]);
                
                $success_message = 'Kullanıcı başarıyla silindi.';
                
                // Listeyi yenile
                $stmt = $pdo->query("
                    SELECT u.*, 
                           COUNT(DISTINCT q.id) as quote_count,
                           COUNT(DISTINCT o.id) as order_count,
                           MAX(u.last_login) as last_activity
                    FROM users u
                    LEFT JOIN quotes q ON u.id = q.created_by
                    LEFT JOIN orders o ON u.id = o.created_by
                    GROUP BY u.id
                    ORDER BY u.created_at DESC
                ");
                $users = $stmt->fetchAll();
            }
        } catch (Exception $e) {
            error_log("User delete error: " . $e->getMessage());
            $error_message = 'Kullanıcı silinirken hata oluştu.';
        }
    }
}

// Kullanıcı durumu değiştirme
if (isset($_POST['toggle_status'])) {
    $user_id = intval($_POST['user_id']);
    $new_status = intval($_POST['new_status']);
    
    if ($user_id === $_SESSION['user_id']) {
        $error_message = 'Kendi hesabınızın durumunu değiştiremezsiniz.';
    } else {
        try {
            $stmt = $pdo->prepare("UPDATE users SET is_active = ?, updated_at = NOW() WHERE id = ?");
            $stmt->execute([$new_status, $user_id]);
            
            logAudit('user_status_changed', 'users', $user_id, null, [
                'new_status' => $new_status,
                'changed_by' => $_SESSION['user_id']
            ]);
            
            $success_message = 'Kullanıcı durumu güncellendi.';
            
            // Listeyi yenile
            $stmt = $pdo->query("
                SELECT u.*, 
                       COUNT(DISTINCT q.id) as quote_count,
                       COUNT(DISTINCT o.id) as order_count,
                       MAX(u.last_login) as last_activity
                FROM users u
                LEFT JOIN quotes q ON u.id = q.created_by
                LEFT JOIN orders o ON u.id = o.created_by
                GROUP BY u.id
                ORDER BY u.created_at DESC
            ");
            $users = $stmt->fetchAll();
        } catch (Exception $e) {
            error_log("User status update error: " . $e->getMessage());
            $error_message = 'Kullanıcı durumu güncellenirken hata oluştu.';
        }
    }
}
?>
<?php require_once 'includes/header.php'; ?>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <?php include 'includes/sidebar.php'; ?>

            <!-- Ana İçerik -->
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2"><i class="fas fa-users"></i> Kullanıcı Yönetimi</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <div class="btn-group me-2">
                            <a href="user_add.php" class="btn btn-sm btn-success">
                                <i class="fas fa-plus"></i> Yeni Kullanıcı
                            </a>
                        </div>
                    </div>
                </div>

                <!-- Flash Messages -->
                <?php if ($success_message): ?>
                    <div class="alert alert-success alert-dismissible fade show">
                        <i class="fas fa-check-circle"></i> <?= $success_message ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <?php if ($error_message): ?>
                    <div class="alert alert-danger alert-dismissible fade show">
                        <i class="fas fa-exclamation-triangle"></i> <?= $error_message ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <!-- İstatistikler -->
                <div class="row mb-4">
                    <?php
                    $total_users = count($users);
                    $active_users = count(array_filter($users, function($u) { return $u['is_active']; }));
                    $admin_users = count(array_filter($users, function($u) { return $u['role'] === 'admin'; }));
                    $recent_logins = count(array_filter($users, function($u) { 
                        return $u['last_login'] && strtotime($u['last_login']) > strtotime('-7 days'); 
                    }));
                    ?>
                    <div class="col-md-3 mb-3">
                        <div class="card bg-primary text-white">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <h4><?= $total_users ?></h4>
                                        <p class="mb-0">Toplam Kullanıcı</p>
                                    </div>
                                    <div class="align-self-center">
                                        <i class="fas fa-users fa-2x"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3 mb-3">
                        <div class="card bg-success text-white">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <h4><?= $active_users ?></h4>
                                        <p class="mb-0">Aktif Kullanıcı</p>
                                    </div>
                                    <div class="align-self-center">
                                        <i class="fas fa-user-check fa-2x"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3 mb-3">
                        <div class="card bg-warning text-white">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <h4><?= $admin_users ?></h4>
                                        <p class="mb-0">Admin Kullanıcı</p>
                                    </div>
                                    <div class="align-self-center">
                                        <i class="fas fa-user-shield fa-2x"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3 mb-3">
                        <div class="card bg-info text-white">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <h4><?= $recent_logins ?></h4>
                                        <p class="mb-0">Son 7 Gün Giriş</p>
                                    </div>
                                    <div class="align-self-center">
                                        <i class="fas fa-sign-in-alt fa-2x"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Kullanıcı Listesi -->
                <div class="card shadow-lg">
                    <div class="card-header">
                        <h5><i class="fas fa-list"></i> Kullanıcı Listesi</h5>
                    </div>
                    <div class="card-body">
                        <?php if (empty($users)): ?>
                            <div class="text-center py-4">
                                <i class="fas fa-users fa-3x text-muted mb-3"></i>
                                <p class="text-muted">Henüz kullanıcı bulunmuyor.</p>
                                <a href="user_add.php" class="btn btn-primary">
                                    <i class="fas fa-plus"></i> İlk Kullanıcıyı Ekle
                                </a>
                            </div>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>Kullanıcı</th>
                                            <th>E-posta</th>
                                            <th>Rol</th>
                                            <th>Durum</th>
                                            <th>İstatistikler</th>
                                            <th>Son Giriş</th>
                                            <th>İşlemler</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($users as $user): ?>
                                        <tr>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <div class="avatar-sm me-3">
                                                        <?php if ($user['id'] === $_SESSION['user_id']): ?>
                                                            <i class="fas fa-star text-warning"></i>
                                                        <?php else: ?>
                                                            <i class="fas fa-user-circle fa-2x text-muted"></i>
                                                        <?php endif; ?>
                                                    </div>
                                                    <div>
                                                        <strong><?= h($user['first_name'] . ' ' . $user['last_name']) ?></strong>
                                                        <br><small class="text-muted">@<?= h($user['username']) ?></small>
                                                        <?php if ($user['id'] === $_SESSION['user_id']): ?>
                                                            <small class="badge bg-warning ms-1">Siz</small>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                            </td>
                                            <td><?= h($user['email']) ?></td>
                                            <td>
                                                <?php
                                                $role_badges = [
                                                    'admin' => 'bg-danger',
                                                    'manager' => 'bg-warning',
                                                    'staff' => 'bg-info'
                                                ];
                                                $role_names = [
                                                    'admin' => 'Admin',
                                                    'manager' => 'Müdür',
                                                    'staff' => 'Personel'
                                                ];
                                                ?>
                                                <span class="badge <?= $role_badges[$user['role']] ?? 'bg-secondary' ?>">
                                                    <?= $role_names[$user['role']] ?? ucfirst($user['role']) ?>
                                                </span>
                                            </td>
                                            <td>
                                                <?php if ($user['is_active']): ?>
                                                    <span class="badge bg-success">Aktif</span>
                                                <?php else: ?>
                                                    <span class="badge bg-danger">Pasif</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <small class="text-muted">
                                                    <i class="fas fa-file-alt"></i> <?= $user['quote_count'] ?> Teklif<br>
                                                    <i class="fas fa-shopping-cart"></i> <?= $user['order_count'] ?> Sipariş
                                                </small>
                                            </td>
                                            <td>
                                                <?php if ($user['last_login']): ?>
                                                    <small class="text-muted">
                                                        <?= date('d.m.Y H:i', strtotime($user['last_login'])) ?>
                                                    </small>
                                                <?php else: ?>
                                                    <small class="text-muted">Hiç giriş yapmamış</small>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <?php if ($user['id'] !== $_SESSION['user_id']): ?>
                                                    <div class="btn-group btn-group-sm">
                                                        <a href="user_edit.php?id=<?= $user['id'] ?>" class="btn btn-outline-primary" title="Düzenle">
                                                            <i class="fas fa-edit"></i>
                                                        </a>
                                                        
                                                        <form method="POST" style="display: inline;" onsubmit="return confirm('Kullanıcı durumunu değiştirmek istediğinizden emin misiniz?')">
                                                            <input type="hidden" name="user_id" value="<?= $user['id'] ?>">
                                                            <input type="hidden" name="new_status" value="<?= $user['is_active'] ? 0 : 1 ?>">
                                                            <button type="submit" name="toggle_status" class="btn btn-outline-<?= $user['is_active'] ? 'warning' : 'success' ?>" 
                                                                    title="<?= $user['is_active'] ? 'Pasif Yap' : 'Aktif Yap' ?>">
                                                                <i class="fas fa-<?= $user['is_active'] ? 'pause' : 'play' ?>"></i>
                                                            </button>
                                                        </form>
                                                        
                                                        <?php if ($user['quote_count'] == 0 && $user['order_count'] == 0): ?>
                                                        <form method="POST" style="display: inline;" onsubmit="return confirm('Bu kullanıcıyı silmek istediğinizden emin misiniz? Bu işlem geri alınamaz!')">
                                                            <input type="hidden" name="user_id" value="<?= $user['id'] ?>">
                                                            <button type="submit" name="delete_user" class="btn btn-outline-danger" title="Sil">
                                                                <i class="fas fa-trash"></i>
                                                            </button>
                                                        </form>
                                                        <?php endif; ?>
                                                    </div>
                                                <?php else: ?>
                                                    <a href="user_edit.php?id=<?= $user['id'] ?>" class="btn btn-sm btn-outline-primary">
                                                        <i class="fas fa-edit"></i> Profil
                                                    </a>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="../assets/js/main.js">    </script>
<?php require_once 'includes/footer.php'; ?>
