<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';

$pageTitle = 'Kullanıcı Ekle';
$currentPage = 'users';

if (!isLoggedIn() || !isAdmin()) {
    redirect('login.php');
}

// Sadece admin kullanıcıları ekleyebilir
if ($_SESSION['user_role'] !== 'admin') {
    addFlashMessage('error', 'Bu sayfaya erişim yetkiniz yok.');
    redirect('dashboard.php');
}

$success_message = '';
$error_message = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $password = trim($_POST['password'] ?? '');
    $password_confirm = trim($_POST['password_confirm'] ?? '');
    $first_name = trim($_POST['first_name'] ?? '');
    $last_name = trim($_POST['last_name'] ?? '');
    $role = $_POST['role'] ?? '';
    $is_active = isset($_POST['is_active']);
    
    $errors = [];
    
    // Validasyon
    if (empty($username)) $errors[] = 'Kullanıcı adı gereklidir.';
    if (empty($email)) $errors[] = 'E-posta adresi gereklidir.';
    if (empty($password)) $errors[] = 'Şifre gereklidir.';
    if (empty($first_name)) $errors[] = 'Ad gereklidir.';
    if (empty($last_name)) $errors[] = 'Soyad gereklidir.';
    if (empty($role)) $errors[] = 'Rol seçimi gereklidir.';
    
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = 'Geçerli bir e-posta adresi giriniz.';
    }
    
    if (strlen($password) < 6) {
        $errors[] = 'Şifre en az 6 karakter olmalıdır.';
    }
    
    if ($password !== $password_confirm) {
        $errors[] = 'Şifreler eşleşmiyor.';
    }
    
    // Kullanıcı adı kontrolü
    if ($username) {
        $stmt = $pdo->prepare("SELECT id FROM users WHERE username = ?");
        $stmt->execute([$username]);
        if ($stmt->fetch()) {
            $errors[] = 'Bu kullanıcı adı zaten kullanılıyor.';
        }
    }
    
    // E-posta kontrolü
    if ($email) {
        $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
        $stmt->execute([$email]);
        if ($stmt->fetch()) {
            $errors[] = 'Bu e-posta adresi zaten kullanılıyor.';
        }
    }
    
    if (empty($errors)) {
        try {
            $password_hash = password_hash($password, PASSWORD_DEFAULT);
            
            $stmt = $pdo->prepare("
                INSERT INTO users (username, email, password_hash, first_name, last_name, role, is_active) 
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ");
            
            $stmt->execute([$username, $email, $password_hash, $first_name, $last_name, $role, $is_active]);
            
            $user_id = $pdo->lastInsertId();
            
            logAudit('user_created', 'users', $user_id, null, [
                'username' => $username,
                'email' => $email,
                'role' => $role,
                'created_by' => $_SESSION['user_id']
            ]);
            
            addFlashMessage('success', 'Kullanıcı başarıyla eklendi.');
            redirect('users.php');
            
        } catch (Exception $e) {
            error_log("User add error: " . $e->getMessage());
            $error_message = 'Bir hata oluştu. Lütfen tekrar deneyiniz.';
        }
    } else {
        $error_message = implode('<br>', $errors);
    }
}
?>
<?php require_once 'includes/header.php'; ?>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <?php include 'includes/sidebar.php'; ?>

            <!-- Ana İçerik -->
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2"><i class="fas fa-user-plus"></i> Yeni Kullanıcı Ekle</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <div class="btn-group me-2">
                            <a href="users.php" class="btn btn-sm btn-outline-secondary">
                                <i class="fas fa-arrow-left"></i> Geri
                            </a>
                        </div>
                    </div>
                </div>

                <!-- Flash Messages -->
                <?php if ($error_message): ?>
                    <div class="alert alert-danger">
                        <i class="fas fa-exclamation-triangle"></i> <?= $error_message ?>
                    </div>
                <?php endif; ?>

                <div class="row">
                    <div class="col-lg-8">
                        <div class="card shadow-lg">
                            <div class="card-header">
                                <h5><i class="fas fa-user-circle"></i> Kullanıcı Bilgileri</h5>
                            </div>
                            <div class="card-body">
                                <form method="POST" action="">
                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label for="first_name" class="form-label">Ad *</label>
                                            <input type="text" class="form-control" id="first_name" name="first_name" 
                                                   value="<?= h($_POST['first_name'] ?? '') ?>" required>
                                        </div>
                                        <div class="col-md-6 mb-3">
                                            <label for="last_name" class="form-label">Soyad *</label>
                                            <input type="text" class="form-control" id="last_name" name="last_name" 
                                                   value="<?= h($_POST['last_name'] ?? '') ?>" required>
                                        </div>
                                    </div>

                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label for="username" class="form-label">Kullanıcı Adı *</label>
                                            <input type="text" class="form-control" id="username" name="username" 
                                                   value="<?= h($_POST['username'] ?? '') ?>" required>
                                            <div class="form-text">Sadece harf, rakam ve alt çizgi kullanın.</div>
                                        </div>
                                        <div class="col-md-6 mb-3">
                                            <label for="email" class="form-label">E-posta *</label>
                                            <input type="email" class="form-control" id="email" name="email" 
                                                   value="<?= h($_POST['email'] ?? '') ?>" required>
                                        </div>
                                    </div>

                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label for="password" class="form-label">Şifre *</label>
                                            <div class="input-group">
                                                <input type="password" class="form-control" id="password" name="password" required>
                                                <button class="btn btn-outline-secondary" type="button" onclick="togglePassword('password')">
                                                    <i class="fas fa-eye" id="password-icon"></i>
                                                </button>
                                            </div>
                                            <div class="form-text">En az 6 karakter olmalıdır.</div>
                                        </div>
                                        <div class="col-md-6 mb-3">
                                            <label for="password_confirm" class="form-label">Şifre Tekrar *</label>
                                            <div class="input-group">
                                                <input type="password" class="form-control" id="password_confirm" name="password_confirm" required>
                                                <button class="btn btn-outline-secondary" type="button" onclick="togglePassword('password_confirm')">
                                                    <i class="fas fa-eye" id="password_confirm-icon"></i>
                                                </button>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label for="role" class="form-label">Rol *</label>
                                            <select class="form-select" id="role" name="role" required>
                                                <option value="">Rol Seçin</option>
                                                <option value="staff" <?= ($_POST['role'] ?? '') === 'staff' ? 'selected' : '' ?>>Personel</option>
                                                <option value="manager" <?= ($_POST['role'] ?? '') === 'manager' ? 'selected' : '' ?>>Müdür</option>
                                                <option value="admin" <?= ($_POST['role'] ?? '') === 'admin' ? 'selected' : '' ?>>Admin</option>
                                            </select>
                                        </div>
                                        <div class="col-md-6 mb-3 d-flex align-items-end">
                                            <div class="form-check">
                                                <input class="form-check-input" type="checkbox" id="is_active" name="is_active" 
                                                       <?= isset($_POST['is_active']) || !isset($_POST['username']) ? 'checked' : '' ?>>
                                                <label class="form-check-label" for="is_active">
                                                    Aktif Kullanıcı
                                                </label>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="d-grid gap-2 d-md-flex justify-content-md-end">
                                        <a href="users.php" class="btn btn-outline-secondary me-md-2">
                                            <i class="fas fa-times"></i> İptal
                                        </a>
                                        <button type="submit" class="btn btn-success">
                                            <i class="fas fa-save"></i> Kullanıcı Ekle
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>

                    <div class="col-lg-4">
                        <div class="card shadow-lg">
                            <div class="card-header">
                                <h5><i class="fas fa-info-circle"></i> Rol Açıklamaları</h5>
                            </div>
                            <div class="card-body">
                                <div class="role-info mb-3">
                                    <h6><i class="fas fa-user text-info"></i> Personel</h6>
                                    <ul class="small text-muted">
                                        <li>Teklif hazırlama</li>
                                        <li>Müşteri görüntüleme</li>
                                        <li>Sipariş takibi</li>
                                    </ul>
                                </div>

                                <div class="role-info mb-3">
                                    <h6><i class="fas fa-user-tie text-warning"></i> Müdür</h6>
                                    <ul class="small text-muted">
                                        <li>Tüm personel yetkileri</li>
                                        <li>Ürün/fiyat yönetimi</li>
                                        <li>Raporlama</li>
                                        <li>Ayarlar</li>
                                    </ul>
                                </div>

                                <div class="role-info">
                                    <h6><i class="fas fa-user-shield text-danger"></i> Admin</h6>
                                    <ul class="small text-muted">
                                        <li>Tüm sistem yetkileri</li>
                                        <li>Kullanıcı yönetimi</li>
                                        <li>Sistem ayarları</li>
                                        <li>Güvenlik ayarları</li>
                                    </ul>
                                </div>
                            </div>
                        </div>

                        <div class="card shadow-lg mt-4">
                            <div class="card-header">
                                <h5><i class="fas fa-shield-alt"></i> Güvenlik İpuçları</h5>
                            </div>
                            <div class="card-body">
                                <ul class="small text-muted">
                                    <li>Güçlü şifre kullanın (en az 8 karakter)</li>
                                    <li>Büyük/küçük harf, rakam ve sembol karışımı</li>
                                    <li>Kişisel bilgiler içermesin</li>
                                    <li>Düzenli olarak şifre değiştirin</li>
                                    <li>Şifreyi kimseyle paylaşmayın</li>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="../assets/js/main.js"></script>
    <script>
        function togglePassword(fieldId) {
            const field = document.getElementById(fieldId);
            const icon = document.getElementById(fieldId + '-icon');
            
            if (field.type === 'password') {
                field.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
            } else {
                field.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
            }
        }

        // Şifre eşleşme kontrolü
        document.getElementById('password_confirm').addEventListener('input', function() {
            const password = document.getElementById('password').value;
            const confirmPassword = this.value;
            
            if (confirmPassword && password !== confirmPassword) {
                this.setCustomValidity('Şifreler eşleşmiyor');
                this.classList.add('is-invalid');
            } else {
                this.setCustomValidity('');
                this.classList.remove('is-invalid');
            }
        });

        // Kullanıcı adı otomatik oluştur
        document.getElementById('first_name').addEventListener('blur', generateUsername);
        document.getElementById('last_name').addEventListener('blur', generateUsername);

        function generateUsername() {
            const firstName = document.getElementById('first_name').value.trim();
            const lastName = document.getElementById('last_name').value.trim();
            const usernameField = document.getElementById('username');
            
            if (!usernameField.value && firstName && lastName) {
                const username = (firstName.toLowerCase() + '.' + lastName.toLowerCase())
                    .replace(/[^a-z.]/g, '')
                    .substring(0, 20);
                usernameField.value = username;
            }
        }
    </script>
<?php require_once 'includes/footer.php'; ?>
