<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Oturum geçersiz.']);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Geçersiz istek metodu.']);
    exit;
}

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    $request_id = intval($input['request_id'] ?? 0);
    $status = trim($input['status'] ?? '');
    
    if (!$request_id || !$status) {
        throw new Exception('Geçersiz parametreler.');
    }
    
    // Geçerli durumlar
    $valid_statuses = ['NEW', 'IN_PROGRESS', 'QUOTE_SENT', 'ACCEPTED', 'REJECTED', 'CANCELLED', 'EXPIRED'];
    
    if (!in_array($status, $valid_statuses)) {
        throw new Exception('Geçersiz durum.');
    }
    
    // Başvuruyu kontrol et
    $stmt = $pdo->prepare("SELECT id, status FROM quote_requests WHERE id = ?");
    $stmt->execute([$request_id]);
    $request = $stmt->fetch();
    
    if (!$request) {
        throw new Exception('Başvuru bulunamadı.');
    }
    
    // Durumu güncelle
    $stmt = $pdo->prepare("UPDATE quote_requests SET status = ?, updated_at = NOW() WHERE id = ?");
    $stmt->execute([$status, $request_id]);
    
    // Audit log
    logAudit('quote_status_updated', 'quote_requests', $request_id, $_SESSION['user_id'], [
        'old_status' => $request['status'],
        'new_status' => $status
    ]);
    
    echo json_encode([
        'success' => true,
        'message' => 'Durum başarıyla güncellendi.',
        'new_status' => $status
    ]);
    
} catch (Exception $e) {
    error_log("Update quote status error: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
?>
