<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';

if (!isLoggedIn()) {
    redirect('login.php');
}

// Sayfalama
$page = intval($_GET['page'] ?? 1);
$limit = 20;
$offset = ($page - 1) * $limit;

// Filtreleme
$status_filter = $_GET['status'] ?? '';
$search = trim($_GET['search'] ?? '');
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';

// Teklif başvurularını al
$where_conditions = ['1=1'];
$params = [];

if ($status_filter) {
    $where_conditions[] = "qr.status = ?";
    $params[] = $status_filter;
}

if ($search) {
    $where_conditions[] = "(c.first_name LIKE ? OR c.last_name LIKE ? OR c.phone LIKE ? OR v.brand LIKE ? OR v.model LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

if ($date_from) {
    $where_conditions[] = "DATE(qr.created_at) >= ?";
    $params[] = $date_from;
}

if ($date_to) {
    $where_conditions[] = "DATE(qr.created_at) <= ?";
    $params[] = $date_to;
}

$where_clause = implode(' AND ', $where_conditions);

// Toplam kayıt sayısı
$count_sql = "
    SELECT COUNT(*) as total 
    FROM quote_requests qr
    JOIN customers c ON qr.customer_id = c.id
    JOIN vehicles v ON qr.vehicle_id = v.id
    WHERE $where_clause
";
$stmt = $pdo->prepare($count_sql);
$stmt->execute($params);
$total_records = $stmt->fetch()['total'];

// Başvuruları getir
$sql = "
    SELECT qr.*, c.first_name, c.last_name, c.phone, c.email,
           v.brand, v.model, v.year, v.fuel_type, v.transmission,
           u.first_name as assigned_first_name, u.last_name as assigned_last_name
    FROM quote_requests qr
    JOIN customers c ON qr.customer_id = c.id
    JOIN vehicles v ON qr.vehicle_id = v.id
    LEFT JOIN users u ON qr.assigned_to = u.id
    WHERE $where_clause
    ORDER BY qr.created_at DESC
    LIMIT $limit OFFSET $offset
";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$quote_requests = $stmt->fetchAll();

$pagination = calculatePagination($total_records, $page, $limit);

// Durum istatistikleri
$stats_sql = "
    SELECT 
        status,
        COUNT(*) as count
    FROM quote_requests 
    GROUP BY status
";
$stmt = $pdo->query($stats_sql);
$status_stats = [];
while ($row = $stmt->fetch()) {
    $status_stats[$row['status']] = $row['count'];
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Teklif Başvuruları - Araç Bakım Servisi</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="../assets/css/style.css" rel="stylesheet">
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <?php include 'includes/sidebar.php'; ?>

            <!-- Ana İçerik -->
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2"><i class="fas fa-clipboard-list"></i> Teklif Başvuruları</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <div class="btn-group me-2">
                            <button type="button" class="btn btn-sm btn-outline-secondary" onclick="location.reload()">
                                <i class="fas fa-sync-alt"></i> Yenile
                            </button>
                        </div>
                    </div>
                </div>

                <!-- Durum İstatistikleri -->
                <div class="row mb-4">
                    <div class="col-md-2 mb-2">
                        <div class="stats-card">
                            <div class="text-center">
                                <div class="stats-number text-primary"><?= $status_stats['NEW'] ?? 0 ?></div>
                                <div class="stats-label">Yeni</div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-2 mb-2">
                        <div class="stats-card">
                            <div class="text-center">
                                <div class="stats-number text-warning"><?= $status_stats['IN_PROGRESS'] ?? 0 ?></div>
                                <div class="stats-label">İşlemde</div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-2 mb-2">
                        <div class="stats-card">
                            <div class="text-center">
                                <div class="stats-number text-info"><?= $status_stats['QUOTE_SENT'] ?? 0 ?></div>
                                <div class="stats-label">Teklif Gönderildi</div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-2 mb-2">
                        <div class="stats-card">
                            <div class="text-center">
                                <div class="stats-number text-success"><?= $status_stats['PAID'] ?? 0 ?></div>
                                <div class="stats-label">Ödendi</div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-2 mb-2">
                        <div class="stats-card">
                            <div class="text-center">
                                <div class="stats-number text-secondary"><?= $status_stats['EXPIRED'] ?? 0 ?></div>
                                <div class="stats-label">Süresi Doldu</div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-2 mb-2">
                        <div class="stats-card">
                            <div class="text-center">
                                <div class="stats-number text-danger"><?= $status_stats['CANCELLED'] ?? 0 ?></div>
                                <div class="stats-label">İptal</div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Filtreler -->
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="GET" class="row g-3">
                            <div class="col-md-3">
                                <label for="search" class="form-label">Arama</label>
                                <input type="text" class="form-control" id="search" name="search" 
                                       value="<?= h($search) ?>" placeholder="Müşteri adı, telefon, araç">
                            </div>
                            <div class="col-md-2">
                                <label for="status" class="form-label">Durum</label>
                                <select class="form-select" id="status" name="status">
                                    <option value="">Tüm Durumlar</option>
                                    <option value="NEW" <?= $status_filter === 'NEW' ? 'selected' : '' ?>>Yeni</option>
                                    <option value="IN_PROGRESS" <?= $status_filter === 'IN_PROGRESS' ? 'selected' : '' ?>>İşlemde</option>
                                    <option value="QUOTE_SENT" <?= $status_filter === 'QUOTE_SENT' ? 'selected' : '' ?>>Teklif Gönderildi</option>
                                    <option value="PAID" <?= $status_filter === 'PAID' ? 'selected' : '' ?>>Ödendi</option>
                                    <option value="EXPIRED" <?= $status_filter === 'EXPIRED' ? 'selected' : '' ?>>Süresi Doldu</option>
                                    <option value="CANCELLED" <?= $status_filter === 'CANCELLED' ? 'selected' : '' ?>>İptal</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label for="date_from" class="form-label">Başlangıç</label>
                                <input type="date" class="form-control" id="date_from" name="date_from" value="<?= h($date_from) ?>">
                            </div>
                            <div class="col-md-2">
                                <label for="date_to" class="form-label">Bitiş</label>
                                <input type="date" class="form-control" id="date_to" name="date_to" value="<?= h($date_to) ?>">
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">&nbsp;</label>
                                <div class="d-grid">
                                    <button type="submit" class="btn btn-primary">
                                        <i class="fas fa-search"></i> Filtrele
                                    </button>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Başvuru Listesi -->
                <div class="card">
                    <div class="card-header">
                        <h5><i class="fas fa-list"></i> Başvuru Listesi (<?= $total_records ?> kayıt)</h5>
                    </div>
                    <div class="card-body">
                        <?php if (empty($quote_requests)): ?>
                            <div class="text-center py-5">
                                <i class="fas fa-clipboard-list fa-3x text-muted mb-3"></i>
                                <p class="text-muted">Henüz teklif başvurusu bulunmuyor.</p>
                            </div>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>ID</th>
                                            <th>Müşteri</th>
                                            <th>İletişim</th>
                                            <th>Araç</th>
                                            <th>Durum</th>
                                            <th>Atanan</th>
                                            <th>Tarih</th>
                                            <th>İşlemler</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($quote_requests as $request): ?>
                                        <tr>
                                            <td><strong>#<?= $request['id'] ?></strong></td>
                                            <td>
                                                <div>
                                                    <strong><?= h($request['first_name'] . ' ' . $request['last_name']) ?></strong>
                                                    <?php if ($request['priority'] !== 'normal'): ?>
                                                        <br><span class="badge bg-<?= $request['priority'] === 'high' ? 'warning' : ($request['priority'] === 'urgent' ? 'danger' : 'info') ?>">
                                                            <?= ucfirst($request['priority']) ?>
                                                        </span>
                                                    <?php endif; ?>
                                                </div>
                                            </td>
                                            <td>
                                                <div>
                                                    <i class="fas fa-phone text-primary"></i> <?= h($request['phone']) ?>
                                                    <?php if ($request['email']): ?>
                                                        <br><i class="fas fa-envelope text-info"></i> <?= h($request['email']) ?>
                                                    <?php endif; ?>
                                                </div>
                                            </td>
                                            <td>
                                                <div>
                                                    <strong><?= h($request['brand'] . ' ' . $request['model']) ?></strong>
                                                    <br><small class="text-muted">
                                                        <?= $request['year'] ?> • <?= h($request['fuel_type']) ?> • <?= h($request['transmission']) ?>
                                                    </small>
                                                </div>
                                            </td>
                                            <td>
                                                <?php
                                                $statusClass = [
                                                    'NEW' => 'badge-new',
                                                    'IN_PROGRESS' => 'badge-in-progress',
                                                    'QUOTE_SENT' => 'badge bg-info',
                                                    'PAID' => 'badge-completed',
                                                    'EXPIRED' => 'badge bg-secondary',
                                                    'CANCELLED' => 'badge-cancelled',
                                                    'PAYMENT_FAILED' => 'badge bg-warning'
                                                ];
                                                $statusText = [
                                                    'NEW' => 'Yeni',
                                                    'IN_PROGRESS' => 'İşlemde',
                                                    'QUOTE_SENT' => 'Teklif Gönderildi',
                                                    'PAID' => 'Ödendi',
                                                    'EXPIRED' => 'Süresi Doldu',
                                                    'CANCELLED' => 'İptal',
                                                    'PAYMENT_FAILED' => 'Ödeme Başarısız'
                                                ];
                                                ?>
                                                <span class="badge <?= $statusClass[$request['status']] ?? 'badge bg-secondary' ?>">
                                                    <?= $statusText[$request['status']] ?? $request['status'] ?>
                                                </span>
                                            </td>
                                            <td>
                                                <?php if ($request['assigned_first_name']): ?>
                                                    <span class="badge bg-primary">
                                                        <?= h($request['assigned_first_name'] . ' ' . $request['assigned_last_name']) ?>
                                                    </span>
                                                <?php else: ?>
                                                    <span class="text-muted">Atanmamış</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <small><?= formatDate($request['created_at'], 'd.m.Y H:i') ?></small>
                                            </td>
                                            <td>
                                                <div class="btn-group btn-group-sm">
                                                    <a href="quote_detail.php?id=<?= $request['id'] ?>" class="btn btn-outline-primary" title="Detay">
                                                        <i class="fas fa-eye"></i>
                                                    </a>
                                                    <?php if ($request['status'] === 'NEW' || $request['status'] === 'IN_PROGRESS'): ?>
                                                    <a href="create_quote.php?request_id=<?= $request['id'] ?>" class="btn btn-outline-success" title="Teklif Hazırla">
                                                        <i class="fas fa-plus"></i>
                                                    </a>
                                                    <?php endif; ?>
                                                    <div class="btn-group btn-group-sm">
                                                        <button type="button" class="btn btn-outline-secondary dropdown-toggle" data-bs-toggle="dropdown">
                                                            <i class="fas fa-ellipsis-v"></i>
                                                        </button>
                                                        <ul class="dropdown-menu">
                                                            <li><a class="dropdown-item" href="tel:<?= $request['phone'] ?>"><i class="fas fa-phone"></i> Ara</a></li>
                                                            <?php if ($request['email']): ?>
                                                            <li><a class="dropdown-item" href="mailto:<?= $request['email'] ?>"><i class="fas fa-envelope"></i> E-posta</a></li>
                                                            <?php endif; ?>
                                                            <li><hr class="dropdown-divider"></li>
                                                            <li><a class="dropdown-item text-warning" href="#" onclick="assignRequest(<?= $request['id'] ?>)"><i class="fas fa-user-plus"></i> Ata</a></li>
                                                            <li><a class="dropdown-item text-danger" href="#" onclick="cancelRequest(<?= $request['id'] ?>)"><i class="fas fa-times"></i> İptal</a></li>
                                                        </ul>
                                                    </div>
                                                </div>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>

                            <!-- Sayfalama -->
                            <?php if ($pagination['total_pages'] > 1): ?>
                                <nav aria-label="Sayfa navigasyonu" class="mt-4">
                                    <ul class="pagination justify-content-center">
                                        <?php if ($pagination['has_previous']): ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?page=<?= $pagination['current_page'] - 1 ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($status_filter) ?>&date_from=<?= urlencode($date_from) ?>&date_to=<?= urlencode($date_to) ?>">
                                                    <i class="fas fa-chevron-left"></i>
                                                </a>
                                            </li>
                                        <?php endif; ?>

                                        <?php for ($i = max(1, $pagination['current_page'] - 2); $i <= min($pagination['total_pages'], $pagination['current_page'] + 2); $i++): ?>
                                            <li class="page-item <?= $i === $pagination['current_page'] ? 'active' : '' ?>">
                                                <a class="page-link" href="?page=<?= $i ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($status_filter) ?>&date_from=<?= urlencode($date_from) ?>&date_to=<?= urlencode($date_to) ?>">
                                                    <?= $i ?>
                                                </a>
                                            </li>
                                        <?php endfor; ?>

                                        <?php if ($pagination['has_next']): ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?page=<?= $pagination['current_page'] + 1 ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($status_filter) ?>&date_from=<?= urlencode($date_from) ?>&date_to=<?= urlencode($date_to) ?>">
                                                    <i class="fas fa-chevron-right"></i>
                                                </a>
                                            </li>
                                        <?php endif; ?>
                                    </ul>
                                </nav>
                            <?php endif; ?>
                        <?php endif; ?>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="../assets/js/main.js"></script>
    <script>
        function assignRequest(id) {
            // Atama modalı veya basit prompt
            const userId = prompt('Kullanıcı ID giriniz:');
            if (userId) {
                fetch('assign_request.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: `request_id=${id}&user_id=${userId}`
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        AracBakim.showAlert('success', data.message);
                        setTimeout(() => location.reload(), 1500);
                    } else {
                        AracBakim.showAlert('danger', data.message);
                    }
                });
            }
        }

        function cancelRequest(id) {
            const reason = prompt('İptal nedeni (isteğe bağlı):');
            if (reason === null) return; // Kullanıcı iptal etti
            
            if (confirm('Bu başvuruyu iptal etmek istediğinizden emin misiniz?')) {
                const formData = new FormData();
                formData.append('request_id', id);
                formData.append('cancel_reason', reason || 'Belirtilmedi');
                
                fetch('cancel_quote.php', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        AracBakim.showAlert('success', data.message);
                        setTimeout(() => location.reload(), 1500);
                    } else {
                        AracBakim.showAlert('danger', data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    AracBakim.showAlert('danger', 'Bir hata oluştu.');
                });
            }
        }

        // Auto-refresh every 2 minutes
        setInterval(() => {
            location.reload();
        }, 120000);
    </script>
</body>
</html>
