<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';

if (!isLoggedIn()) {
    redirect('login.php');
}

// Sayfalama
$page = intval($_GET['page'] ?? 1);
$limit = 20;
$offset = ($page - 1) * $limit;

// Filtreleme
$status_filter = $_GET['status'] ?? '';
$search = trim($_GET['search'] ?? '');
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';

// Teklif başvurularını al
$where_conditions = ['1=1'];
$params = [];

if ($status_filter) {
    $where_conditions[] = "qr.status = ?";
    $params[] = $status_filter;
}

if ($search) {
    $where_conditions[] = "(c.first_name LIKE ? OR c.last_name LIKE ? OR c.phone LIKE ? OR v.brand LIKE ? OR v.model LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

if ($date_from) {
    $where_conditions[] = "DATE(qr.created_at) >= ?";
    $params[] = $date_from;
}

if ($date_to) {
    $where_conditions[] = "DATE(qr.created_at) <= ?";
    $params[] = $date_to;
}

$where_clause = implode(' AND ', $where_conditions);

try {
    // Toplam kayıt sayısı
    $count_sql = "
        SELECT COUNT(*) as total
        FROM quote_requests qr
        JOIN customers c ON qr.customer_id = c.id
        JOIN vehicles v ON qr.vehicle_id = v.id
        WHERE $where_clause
    ";
    $stmt = $pdo->prepare($count_sql);
    $stmt->execute($params);
    $total_records = $stmt->fetch()['total'];
    $total_pages = ceil($total_records / $limit);

    // Başvuruları al
    $sql = "
        SELECT qr.*, c.first_name, c.last_name, c.phone, c.email,
               v.brand, v.model, v.year, v.plate
        FROM quote_requests qr
        JOIN customers c ON qr.customer_id = c.id
        JOIN vehicles v ON qr.vehicle_id = v.id
        WHERE $where_clause
        ORDER BY qr.created_at DESC
        LIMIT $limit OFFSET $offset
    ";
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $requests = $stmt->fetchAll();

    // İstatistikler
    $stats_sql = "
        SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN status = 'NEW' THEN 1 ELSE 0 END) as new_count,
            SUM(CASE WHEN status = 'QUOTE_SENT' THEN 1 ELSE 0 END) as sent_count,
            SUM(CASE WHEN status = 'ACCEPTED' THEN 1 ELSE 0 END) as accepted_count,
            SUM(CASE WHEN status = 'REJECTED' THEN 1 ELSE 0 END) as rejected_count
        FROM quote_requests qr
        JOIN customers c ON qr.customer_id = c.id
        JOIN vehicles v ON qr.vehicle_id = v.id
        WHERE $where_clause
    ";
    $stmt = $pdo->prepare($stats_sql);
    $stmt->execute($params);
    $stats = $stmt->fetch();

} catch (Exception $e) {
    error_log("Quotes page error: " . $e->getMessage());
    $requests = [];
    $total_records = 0;
    $total_pages = 0;
    $stats = ['total' => 0, 'new_count' => 0, 'sent_count' => 0, 'accepted_count' => 0, 'rejected_count' => 0];
}

// Bekleyen başvuru sayısı sidebar için
try {
    $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM quote_requests WHERE status = 'NEW'");
    $stmt->execute();
    $pendingQuotes = $stmt->fetch()['count'];
} catch (Exception $e) {
    $pendingQuotes = 0;
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Teklif Başvuruları - Araç Bakım Servisi</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Open+Sans:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Nunito:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link href="../assets/css/style.css" rel="stylesheet">
</head>
<body class="nice-body">
    <!-- NiceAdmin Sidebar -->
    <aside id="sidebar" class="nice-sidebar">
        <ul class="sidebar-nav" id="sidebar-nav">
            <li class="nav-item">
                <div class="text-center mb-4 p-3">
                    <i class="fas fa-car text-primary fa-2x"></i>
                    <h6 class="text-primary mt-2 mb-0">Araç Bakım</h6>
                </div>
            </li>
            
            <li class="nav-item">
                <a class="nav-link" href="dashboard.php">
                    <i class="fas fa-tachometer-alt"></i>
                    <span>Dashboard</span>
                </a>
            </li>
            
            <li class="nav-item">
                <a class="nav-link active" href="quotes.php">
                    <i class="fas fa-clipboard-list"></i>
                    <span>Teklif Başvuruları</span>
                    <?php if ($pendingQuotes > 0): ?>
                        <span class="nice-badge nice-badge-danger ms-2"><?= $pendingQuotes ?></span>
                    <?php endif; ?>
                </a>
            </li>
            
            <li class="nav-item">
                <a class="nav-link" href="orders.php">
                    <i class="fas fa-shopping-cart"></i>
                    <span>Siparişler</span>
                </a>
            </li>
            
            <li class="nav-item">
                <a class="nav-link" href="customers.php">
                    <i class="fas fa-users"></i>
                    <span>Müşteriler</span>
                </a>
            </li>
            
            <li class="nav-item">
                <a class="nav-link" href="products.php">
                    <i class="fas fa-box"></i>
                    <span>Ürün/Hizmetler</span>
                </a>
            </li>
            
            <li class="nav-item">
                <a class="nav-link" href="reports.php">
                    <i class="fas fa-chart-bar"></i>
                    <span>Raporlar</span>
                </a>
            </li>
            
            <?php if (isAdmin()): ?>
            <li class="nav-item">
                <a class="nav-link" href="settings.php">
                    <i class="fas fa-cog"></i>
                    <span>Ayarlar</span>
                </a>
            </li>
            
            <li class="nav-item">
                <a class="nav-link" href="users.php">
                    <i class="fas fa-user-cog"></i>
                    <span>Kullanıcılar</span>
                </a>
            </li>
            <?php endif; ?>
            
            <li class="nav-item mt-4">
                <a class="nav-link" href="logout.php">
                    <i class="fas fa-sign-out-alt"></i>
                    <span>Çıkış</span>
                </a>
            </li>
        </ul>
    </aside>

    <!-- NiceAdmin Header -->
    <header id="header" class="nice-header">
        <div class="d-flex align-items-center justify-content-between">
            <div class="logo d-flex align-items-center">
                <span class="d-none d-lg-block">Teklif Başvuruları</span>
            </div>
            <nav class="header-nav ms-auto">
                <ul class="d-flex align-items-center">
                    <li class="nav-item dropdown pe-3">
                        <a class="nav-link nav-profile d-flex align-items-center pe-0" href="#" data-bs-toggle="dropdown">
                            <i class="fas fa-user-circle fa-2x text-primary"></i>
                            <span class="d-none d-md-block dropdown-toggle ps-2"><?= h($_SESSION['user_name']) ?></span>
                        </a>
                        <ul class="dropdown-menu dropdown-menu-end dropdown-menu-arrow profile">
                            <li class="dropdown-header">
                                <h6><?= h($_SESSION['user_name']) ?></h6>
                                <span><?= ucfirst($_SESSION['user_role']) ?></span>
                            </li>
                            <li><hr class="dropdown-divider"></li>
                            <li>
                                <a class="dropdown-item d-flex align-items-center" href="profile.php">
                                    <i class="fas fa-user"></i>
                                    <span>Profil</span>
                                </a>
                            </li>
                            <li><hr class="dropdown-divider"></li>
                            <li>
                                <a class="dropdown-item d-flex align-items-center" href="logout.php">
                                    <i class="fas fa-sign-out-alt"></i>
                                    <span>Çıkış</span>
                                </a>
                            </li>
                        </ul>
                    </li>
                </ul>
            </nav>
        </div>
    </header>

    <!-- NiceAdmin Main Content -->
    <main id="main" class="nice-main">
        <div class="pagetitle">
            <h1>Teklif Başvuruları</h1>
            <nav>
                <ol class="breadcrumb">
                    <li class="breadcrumb-item"><a href="dashboard.php">Ana Sayfa</a></li>
                    <li class="breadcrumb-item active">Teklif Başvuruları</li>
                </ol>
            </nav>
        </div>

        <section class="section">
            <!-- İstatistik Kartları -->
            <div class="row mb-4">
                <div class="col-lg-3 col-md-6">
                    <div class="nice-card info-card sales-card">
                        <div class="card-body">
                            <h5 class="card-title">Toplam Başvuru</h5>
                            <div class="d-flex align-items-center">
                                <div class="card-icon rounded-circle d-flex align-items-center justify-content-center">
                                    <i class="fas fa-clipboard-list"></i>
                                </div>
                                <div class="ps-3">
                                    <h6><?= $stats['total'] ?></h6>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="col-lg-3 col-md-6">
                    <div class="nice-card info-card revenue-card">
                        <div class="card-body">
                            <h5 class="card-title">Yeni Başvuru</h5>
                            <div class="d-flex align-items-center">
                                <div class="card-icon rounded-circle d-flex align-items-center justify-content-center">
                                    <i class="fas fa-plus-circle"></i>
                                </div>
                                <div class="ps-3">
                                    <h6><?= $stats['new_count'] ?></h6>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="col-lg-3 col-md-6">
                    <div class="nice-card info-card customers-card">
                        <div class="card-body">
                            <h5 class="card-title">Gönderildi</h5>
                            <div class="d-flex align-items-center">
                                <div class="card-icon rounded-circle d-flex align-items-center justify-content-center">
                                    <i class="fas fa-paper-plane"></i>
                                </div>
                                <div class="ps-3">
                                    <h6><?= $stats['sent_count'] ?></h6>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="col-lg-3 col-md-6">
                    <div class="nice-card info-card">
                        <div class="card-body">
                            <h5 class="card-title">Kabul Edildi</h5>
                            <div class="d-flex align-items-center">
                                <div class="card-icon rounded-circle d-flex align-items-center justify-content-center" style="color: var(--nice-success); background: rgba(25, 135, 84, 0.1);">
                                    <i class="fas fa-check-circle"></i>
                                </div>
                                <div class="ps-3">
                                    <h6><?= $stats['accepted_count'] ?></h6>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Filtreler -->
            <div class="nice-card">
                <div class="card-body">
                    <h5 class="card-title">Filtreler</h5>
                    <form method="GET" class="row g-3">
                        <div class="col-md-3">
                            <label for="search" class="form-label">Arama</label>
                            <input type="text" class="nice-form-control" id="search" name="search" 
                                   value="<?= h($search) ?>" placeholder="Ad, soyad, telefon, araç...">
                        </div>
                        <div class="col-md-2">
                            <label for="status" class="form-label">Durum</label>
                            <select class="nice-form-control" id="status" name="status">
                                <option value="">Tümü</option>
                                <option value="NEW" <?= $status_filter === 'NEW' ? 'selected' : '' ?>>Yeni</option>
                                <option value="QUOTE_SENT" <?= $status_filter === 'QUOTE_SENT' ? 'selected' : '' ?>>Teklif Gönderildi</option>
                                <option value="ACCEPTED" <?= $status_filter === 'ACCEPTED' ? 'selected' : '' ?>>Kabul Edildi</option>
                                <option value="REJECTED" <?= $status_filter === 'REJECTED' ? 'selected' : '' ?>>Reddedildi</option>
                                <option value="CANCELLED" <?= $status_filter === 'CANCELLED' ? 'selected' : '' ?>>İptal Edildi</option>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label for="date_from" class="form-label">Başlangıç</label>
                            <input type="date" class="nice-form-control" id="date_from" name="date_from" value="<?= h($date_from) ?>">
                        </div>
                        <div class="col-md-2">
                            <label for="date_to" class="form-label">Bitiş</label>
                            <input type="date" class="nice-form-control" id="date_to" name="date_to" value="<?= h($date_to) ?>">
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">&nbsp;</label>
                            <div class="d-flex gap-2">
                                <button type="submit" class="nice-btn nice-btn-primary">
                                    <i class="fas fa-search"></i> Filtrele
                                </button>
                                <a href="quotes.php" class="nice-btn nice-btn-outline-primary">
                                    <i class="fas fa-times"></i> Temizle
                                </a>
                            </div>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Başvurular Tablosu -->
            <div class="nice-card">
                <div class="card-body">
                    <h5 class="card-title">
                        Başvurular 
                        <span class="nice-badge nice-badge-primary"><?= $total_records ?></span>
                    </h5>
                    
                    <?php if (empty($requests)): ?>
                        <div class="alert alert-info">
                            <i class="fas fa-info-circle"></i> Henüz başvuru bulunmuyor.
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="nice-table table-hover">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>Müşteri</th>
                                        <th>İletişim</th>
                                        <th>Araç</th>
                                        <th>Hizmet Türü</th>
                                        <th>Durum</th>
                                        <th>Tarih</th>
                                        <th>İşlemler</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($requests as $request): ?>
                                    <tr>
                                        <td><strong>#<?= $request['id'] ?></strong></td>
                                        <td>
                                            <strong><?= h($request['first_name'] . ' ' . $request['last_name']) ?></strong>
                                        </td>
                                        <td>
                                            <div class="small">
                                                <i class="fas fa-phone text-primary"></i> <?= h($request['phone']) ?><br>
                                                <?php if ($request['email']): ?>
                                                <i class="fas fa-envelope text-info"></i> <?= h($request['email']) ?>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                        <td>
                                            <strong><?= h($request['brand'] . ' ' . $request['model']) ?></strong><br>
                                            <small class="text-muted">
                                                <?= $request['year'] ?>
                                                <?php if ($request['plate']): ?>
                                                    • <?= h($request['plate']) ?>
                                                <?php endif; ?>
                                            </small>
                                        </td>
                                        <td>
                                            <?php if ($request['service_type']): ?>
                                                <span class="nice-badge nice-badge-info"><?= h($request['service_type']) ?></span>
                                            <?php else: ?>
                                                <span class="text-muted">Belirtilmemiş</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php
                                            $statusClass = match($request['status']) {
                                                'NEW' => 'nice-badge-warning',
                                                'QUOTE_SENT' => 'nice-badge-info',
                                                'ACCEPTED' => 'nice-badge-success',
                                                'REJECTED' => 'nice-badge-danger',
                                                'CANCELLED' => 'nice-badge-secondary',
                                                default => 'nice-badge-secondary'
                                            };
                                            $statusText = match($request['status']) {
                                                'NEW' => 'Yeni',
                                                'QUOTE_SENT' => 'Teklif Gönderildi',
                                                'ACCEPTED' => 'Kabul Edildi',
                                                'REJECTED' => 'Reddedildi',
                                                'CANCELLED' => 'İptal Edildi',
                                                default => $request['status']
                                            };
                                            ?>
                                            <span class="nice-badge <?= $statusClass ?>"><?= $statusText ?></span>
                                        </td>
                                        <td>
                                            <small><?= formatDate($request['created_at'], 'd.m.Y H:i') ?></small>
                                        </td>
                                        <td>
                                            <div class="btn-group" role="group">
                                                <button type="button" class="nice-btn nice-btn-primary btn-sm dropdown-toggle" data-bs-toggle="dropdown">
                                                    <i class="fas fa-cog"></i>
                                                </button>
                                                <ul class="dropdown-menu">
                                                    <li>
                                                        <a class="dropdown-item" href="quote_detail.php?id=<?= $request['id'] ?>">
                                                            <i class="fas fa-eye"></i> Detay
                                                        </a>
                                                    </li>
                                                    <?php if ($request['status'] === 'NEW'): ?>
                                                    <li>
                                                        <a class="dropdown-item" href="create_quote.php?request_id=<?= $request['id'] ?>">
                                                            <i class="fas fa-plus"></i> Teklif Hazırla
                                                        </a>
                                                    </li>
                                                    <?php endif; ?>
                                                    <?php if (in_array($request['status'], ['NEW', 'QUOTE_SENT'])): ?>
                                                    <li>
                                                        <a class="dropdown-item text-danger" href="#" onclick="cancelRequest(<?= $request['id'] ?>)">
                                                            <i class="fas fa-times"></i> İptal
                                                        </a>
                                                    </li>
                                                    <?php endif; ?>
                                                </ul>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>

                        <!-- Sayfalama -->
                        <?php if ($total_pages > 1): ?>
                        <nav aria-label="Sayfa navigasyonu" class="mt-4">
                            <ul class="pagination justify-content-center">
                                <?php if ($page > 1): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $page - 1])) ?>">
                                        <i class="fas fa-chevron-left"></i>
                                    </a>
                                </li>
                                <?php endif; ?>
                                
                                <?php for ($i = max(1, $page - 2); $i <= min($total_pages, $page + 2); $i++): ?>
                                <li class="page-item <?= $i === $page ? 'active' : '' ?>">
                                    <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $i])) ?>"><?= $i ?></a>
                                </li>
                                <?php endfor; ?>
                                
                                <?php if ($page < $total_pages): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $page + 1])) ?>">
                                        <i class="fas fa-chevron-right"></i>
                                    </a>
                                </li>
                                <?php endif; ?>
                            </ul>
                        </nav>
                        <?php endif; ?>
                    <?php endif; ?>
                </div>
            </div>
        </section>
    </main>

    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function cancelRequest(id) {
            const reason = prompt('İptal nedeni (isteğe bağlı):');
            if (reason === null) return; // Kullanıcı iptal etti
            
            if (confirm('Bu başvuruyu iptal etmek istediğinizden emin misiniz?')) {
                const formData = new FormData();
                formData.append('request_id', id);
                formData.append('cancel_reason', reason || 'Belirtilmedi');
                
                fetch('cancel_quote.php', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        alert('Başvuru başarıyla iptal edildi.');
                        location.reload();
                    } else {
                        alert('Hata: ' + data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Bir hata oluştu.');
                });
            }
        }
    </script>
</body>
</html>
