<?php
// Teklif ile ilgili yardımcı fonksiyonlar

function generateQuotePDF($quote_id, $pdo) {
    try {
        // Teklif bilgilerini al
        $stmt = $pdo->prepare("
            SELECT q.*, c.first_name, c.last_name, c.email, c.phone, 
                   v.brand, v.model, v.year, v.plate
            FROM quotes q
            JOIN quote_requests qr ON q.quote_request_id = qr.id
            JOIN customers c ON qr.customer_id = c.id
            JOIN vehicles v ON qr.vehicle_id = v.id
            WHERE q.id = ?
        ");
        $stmt->execute([$quote_id]);
        $quote = $stmt->fetch();
        
        if (!$quote) return null;
        
        // Teklif kalemlerini al
        $stmt = $pdo->prepare("
            SELECT qi.*, p.name as product_name, p.type
            FROM quote_items qi
            LEFT JOIN products p ON qi.product_id = p.id
            WHERE qi.quote_id = ?
        ");
        $stmt->execute([$quote_id]);
        $items = $stmt->fetchAll();
        
        // Basit HTML PDF oluştur
        $html = generateQuoteHTML($quote, $items);
        
        // PDF dosya adı
        $filename = 'quote_' . $quote['quote_number'] . '.html';
        $filepath = '../uploads/quotes/' . $filename;
        
        // Klasör yoksa oluştur
        if (!is_dir('../uploads/quotes/')) {
            mkdir('../uploads/quotes/', 0755, true);
        }
        
        // HTML dosyası olarak kaydet (gerçek projede PDF'e çevir)
        file_put_contents($filepath, $html);
        
        return $filename;
        
    } catch (Exception $e) {
        error_log("PDF generation error: " . $e->getMessage());
        return null;
    }
}

function generateQuoteHTML($quote, $items) {
    $html = '
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="UTF-8">
        <title>Teklif - ' . $quote['quote_number'] . '</title>
        <style>
            body { font-family: Arial, sans-serif; margin: 20px; }
            .header { text-align: center; margin-bottom: 30px; }
            .info-table { width: 100%; margin-bottom: 20px; }
            .info-table td { padding: 5px; border-bottom: 1px solid #ddd; }
            .items-table { width: 100%; border-collapse: collapse; margin-bottom: 20px; }
            .items-table th, .items-table td { padding: 8px; border: 1px solid #ddd; text-align: left; }
            .items-table th { background-color: #f5f5f5; }
            .total-section { text-align: right; margin-top: 20px; }
            .total-row { margin: 5px 0; }
        </style>
    </head>
    <body>
        <div class="header">
            <h1>ARAÇ BAKIM SERVİSİ</h1>
            <h2>TEKLİF FORMU</h2>
            <p>Teklif No: ' . $quote['quote_number'] . '</p>
            <p>Tarih: ' . date('d.m.Y', strtotime($quote['created_at'])) . '</p>
        </div>
        
        <table class="info-table">
            <tr>
                <td><strong>Müşteri:</strong></td>
                <td>' . $quote['first_name'] . ' ' . $quote['last_name'] . '</td>
            </tr>
            <tr>
                <td><strong>E-posta:</strong></td>
                <td>' . $quote['email'] . '</td>
            </tr>
            <tr>
                <td><strong>Telefon:</strong></td>
                <td>' . $quote['phone'] . '</td>
            </tr>
            <tr>
                <td><strong>Araç:</strong></td>
                <td>' . $quote['brand'] . ' ' . $quote['model'] . ' (' . $quote['year'] . ') - ' . $quote['plate'] . '</td>
            </tr>
        </table>
        
        <table class="items-table">
            <thead>
                <tr>
                    <th>Ürün/Hizmet</th>
                    <th>Miktar</th>
                    <th>Birim Fiyat</th>
                    <th>Toplam</th>
                </tr>
            </thead>
            <tbody>';
    
    $subtotal = 0;
    foreach ($items as $item) {
        $item_total = $item['quantity'] * $item['unit_price'];
        $subtotal += $item_total;
        
        $html .= '
                <tr>
                    <td>' . ($item['product_name'] ?: $item['name']) . '</td>
                    <td>' . $item['quantity'] . '</td>
                    <td>' . number_format($item['unit_price'], 2) . ' ₺</td>
                    <td>' . number_format($item_total, 2) . ' ₺</td>
                </tr>';
    }
    
    $discount = $subtotal * ($quote['discount_percentage'] / 100);
    $after_discount = $subtotal - $discount;
    $tax = $after_discount * $quote['tax_rate'];
    $total = $after_discount + $tax;
    
    $html .= '
            </tbody>
        </table>
        
        <div class="total-section">
            <div class="total-row"><strong>Ara Toplam: ' . number_format($subtotal, 2) . ' ₺</strong></div>
            <div class="total-row">İndirim (%' . $quote['discount_percentage'] . '): ' . number_format($discount, 2) . ' ₺</div>
            <div class="total-row">KDV (%' . ($quote['tax_rate'] * 100) . '): ' . number_format($tax, 2) . ' ₺</div>
            <div class="total-row"><h3>GENEL TOPLAM: ' . number_format($total, 2) . ' ₺</h3></div>
        </div>
        
        <div style="margin-top: 30px;">
            <p><strong>Şartlar ve Koşullar:</strong></p>
            <p>' . nl2br($quote['terms_conditions']) . '</p>
        </div>
        
        <div style="margin-top: 30px; text-align: center;">
            <p>Bu teklifi onaylamak için aşağıdaki linke tıklayınız:</p>
            <p><a href="http://localhost:8000/view_quote.php?token=' . $quote['quote_token'] . '">Teklifi Görüntüle ve Onayla</a></p>
        </div>
    </body>
    </html>';
    
    return $html;
}

function sendQuoteNotification($customer, $quote_link, $pdf_filename, $quote = null) {
    try {
        // E-posta fonksiyonlarını dahil et (eğer daha önce dahil edilmemişse)
        if (!function_exists('sendQuoteEmail')) {
            require_once '../includes/email_functions.php';
        }
        
        // Eğer quote bilgisi verilmemişse temel bilgileri oluştur
        if (!$quote) {
            $quote = [
                'quote_number' => 'TKL-' . date('YmdHis'),
                'created_at' => date('Y-m-d H:i:s'),
                'valid_until' => date('Y-m-d H:i:s', strtotime('+48 hours')),
                'total_amount' => 0
            ];
        }
        
        // Müşteri bilgilerini kontrol et
        if (!$customer || !is_array($customer)) {
            error_log("Invalid customer data in sendQuoteNotification: " . print_r($customer, true));
            return false;
        }
        
        // Gerçek e-posta gönder
        $emailSent = sendQuoteEmail($customer, $quote, $quote_link, $pdf_filename);
        
        if ($emailSent) {
            error_log("Quote email sent successfully to: " . $customer['email']);
            
            // Ek olarak SMS de gönderilebilir (isteğe bağlı)
            if (isset($customer['phone']) && !empty($customer['phone'])) {
                sendQuoteSMS($customer, $quote_link);
            }
            
            return true;
        } else {
            error_log("Failed to send quote email to: " . $customer['email']);
            return false;
        }
        
    } catch (Exception $e) {
        error_log("Quote notification error: " . $e->getMessage());
        return false;
    }
}

/**
 * SMS bildirimi gönder (isteğe bağlı)
 */
function sendQuoteSMS($customer, $quote_link) {
    try {
        // SMS API entegrasyonu burada yapılabilir
        // Örnek: Netgsm, İletimerkezi, Twilio vb.
        
        $phone = $customer['phone'];
        $message = "Sayın " . $customer['first_name'] . ", aracınız için teklif hazır! ";
        $message .= "Görüntülemek için: " . $quote_link . " ";
        $message .= "- " . SITE_NAME;
        
        // Şimdilik log olarak kaydet
        error_log("SMS would be sent to $phone: $message");
        
        // Gerçek SMS gönderimi için API çağrısı yapılacak
        // return sendSMSViaAPI($phone, $message);
        
        return true;
        
    } catch (Exception $e) {
        error_log("SMS sending error: " . $e->getMessage());
        return false;
    }
}
?>
