<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';

$pageTitle = 'Teklif Detayı';
$currentPage = 'quotes';

if (!isLoggedIn()) {
    redirect('login.php');
}

$quote_request_id = intval($_GET['id'] ?? 0);

if (!$quote_request_id) {
    addFlashMessage('danger', 'Geçersiz teklif başvuru ID.');
    redirect('quotes.php');
}

try {
    // Teklif başvuru detaylarını al
    $stmt = $pdo->prepare("
        SELECT qr.*, c.first_name, c.last_name, c.phone, c.email, c.contact_address,
               v.brand, v.model, v.year, v.fuel_type, v.transmission, v.plate, v.chassis_number, v.mileage,
               u.first_name as assigned_first_name, u.last_name as assigned_last_name
        FROM quote_requests qr
        JOIN customers c ON qr.customer_id = c.id
        JOIN vehicles v ON qr.vehicle_id = v.id
        LEFT JOIN users u ON qr.assigned_to = u.id
        WHERE qr.id = ?
    ");
    $stmt->execute([$quote_request_id]);
    $request = $stmt->fetch();
    
    if (!$request) {
        addFlashMessage('danger', 'Teklif başvurusu bulunamadı.');
        redirect('quotes.php');
    }
    
    // Varsa hazırlanmış teklifi al
    $stmt = $pdo->prepare("
        SELECT q.*, qi.name as item_name, qi.description, qi.quantity, qi.unit_price, qi.line_total
        FROM quotes q
        LEFT JOIN quote_items qi ON q.id = qi.quote_id
        WHERE q.quote_request_id = ?
        ORDER BY qi.sort_order
    ");
    $stmt->execute([$quote_request_id]);
    $quote_data = $stmt->fetchAll();
    
    $quote = null;
    $quote_items = [];
    
    if (!empty($quote_data)) {
        $quote = [
            'id' => $quote_data[0]['id'],
            'title' => $quote_data[0]['title'],
            'subtotal' => $quote_data[0]['subtotal'],
            'tax_rate' => $quote_data[0]['tax_rate'],
            'tax_amount' => $quote_data[0]['tax_amount'],
            'discount_rate' => $quote_data[0]['discount_rate'],
            'discount_amount' => $quote_data[0]['discount_amount'],
            'total_amount' => $quote_data[0]['total_amount'],
            'notes' => $quote_data[0]['notes'],
            'valid_until' => $quote_data[0]['valid_until'],
            'quote_token' => $quote_data[0]['quote_token'],
            'quote_link' => $quote_data[0]['quote_link'],
            'status' => $quote_data[0]['status'],
            'created_at' => $quote_data[0]['created_at']
        ];
        
        foreach ($quote_data as $item) {
            if ($item['item_name']) {
                $quote_items[] = [
                    'name' => $item['item_name'],
                    'description' => $item['description'],
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                    'line_total' => $item['line_total']
                ];
            }
        }
    }
    
    // Hizmet türü çevirisi
    $service_types = [
        'periyodik_bakim' => 'Periyodik Bakım',
        'motor_onarim' => 'Motor Onarımı',
        'kaporta_boya' => 'Kaporta & Boya',
        'genel_onarim' => 'Genel Onarım'
    ];
    
    // Durum çevirisi
    $status_labels = [
        'NEW' => ['text' => 'Yeni', 'class' => 'nice-badge-warning'],
        'IN_PROGRESS' => ['text' => 'İşlemde', 'class' => 'nice-badge-info'],
        'QUOTE_SENT' => ['text' => 'Teklif Gönderildi', 'class' => 'nice-badge-primary'],
        'ACCEPTED' => ['text' => 'Kabul Edildi', 'class' => 'nice-badge-success'],
        'REJECTED' => ['text' => 'Reddedildi', 'class' => 'nice-badge-danger'],
        'CANCELLED' => ['text' => 'İptal Edildi', 'class' => 'nice-badge-secondary'],
        'EXPIRED' => ['text' => 'Süresi Doldu', 'class' => 'nice-badge-dark']
    ];
    
} catch (Exception $e) {
    error_log("Quote detail error: " . $e->getMessage());
    addFlashMessage('danger', 'Bir hata oluştu.');
    redirect('quotes.php');
}
?>
<?php require_once 'includes/header.php'; ?>

<section class="section">
    <div class="pagetitle">
        <h1>Teklif Detayı</h1>
        <nav>
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="dashboard.php">Ana Sayfa</a></li>
                <li class="breadcrumb-item"><a href="quotes.php">Teklif Başvuruları</a></li>
                <li class="breadcrumb-item active">Teklif Detayı</li>
            </ol>
        </nav>
    </div>

    <div class="row">
        <div class="col-lg-8">
            <!-- Başvuru Bilgileri -->
            <div class="nice-card mb-4">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center mb-3">
                        <h5 class="card-title mb-0">
                            <i class="fas fa-file-alt me-2"></i>Başvuru Bilgileri
                        </h5>
                        <span class="nice-badge <?= $status_labels[$request['status']]['class'] ?? 'nice-badge-secondary' ?>">
                            <?= $status_labels[$request['status']]['text'] ?? $request['status'] ?>
                        </span>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <h6 class="text-primary mb-3">
                                <i class="fas fa-user me-2"></i>Müşteri Bilgileri
                            </h6>
                            <table class="table table-borderless table-sm">
                                <tr>
                                    <td class="text-muted" style="width: 120px;">Ad Soyad:</td>
                                    <td><strong><?= h($request['first_name'] . ' ' . $request['last_name']) ?></strong></td>
                                </tr>
                                <tr>
                                    <td class="text-muted">Telefon:</td>
                                    <td><?= h($request['phone']) ?></td>
                                </tr>
                                <?php if ($request['email']): ?>
                                <tr>
                                    <td class="text-muted">E-posta:</td>
                                    <td><?= h($request['email']) ?></td>
                                </tr>
                                <?php endif; ?>
                                <?php if ($request['contact_address']): ?>
                                <tr>
                                    <td class="text-muted">Adres:</td>
                                    <td><?= h($request['contact_address']) ?></td>
                                </tr>
                                <?php endif; ?>
                            </table>
                        </div>
                        
                        <div class="col-md-6">
                            <h6 class="text-primary mb-3">
                                <i class="fas fa-car me-2"></i>Araç Bilgileri
                            </h6>
                            <table class="table table-borderless table-sm">
                                <tr>
                                    <td class="text-muted" style="width: 120px;">Araç:</td>
                                    <td><strong><?= h($request['brand'] . ' ' . $request['model'] . ' (' . $request['year'] . ')') ?></strong></td>
                                </tr>
                                <?php if ($request['plate']): ?>
                                <tr>
                                    <td class="text-muted">Plaka:</td>
                                    <td><?= h($request['plate']) ?></td>
                                </tr>
                                <?php endif; ?>
                                <?php if ($request['chassis_number']): ?>
                                <tr>
                                    <td class="text-muted">Şase No:</td>
                                    <td><?= h($request['chassis_number']) ?></td>
                                </tr>
                                <?php endif; ?>
                                <?php if ($request['fuel_type']): ?>
                                <tr>
                                    <td class="text-muted">Yakıt Türü:</td>
                                    <td><?= ucfirst($request['fuel_type']) ?></td>
                                </tr>
                                <?php endif; ?>
                                <?php if ($request['transmission']): ?>
                                <tr>
                                    <td class="text-muted">Vites:</td>
                                    <td><?= ucfirst($request['transmission']) ?></td>
                                </tr>
                                <?php endif; ?>
                                <?php if ($request['mileage']): ?>
                                <tr>
                                    <td class="text-muted">Kilometre:</td>
                                    <td><?= number_format($request['mileage']) ?> km</td>
                                </tr>
                                <?php endif; ?>
                            </table>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Hizmet Detayları -->
            <div class="nice-card mb-4">
                <div class="card-body">
                    <h5 class="card-title">
                        <i class="fas fa-wrench me-2"></i>Hizmet Detayları
                    </h5>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <p><strong>Hizmet Türü:</strong></p>
                            <span class="nice-badge nice-badge-info">
                                <?= h($service_types[$request['service_type']] ?? $request['service_type']) ?>
                            </span>
                        </div>
                        
                        <div class="col-md-6">
                            <p><strong>Başvuru Tarihi:</strong></p>
                            <p class="text-muted"><?= formatDate($request['created_at'], 'd.m.Y H:i') ?></p>
                        </div>
                    </div>
                    
                    <?php if ($request['problem_description']): ?>
                    <div class="mt-3">
                        <p><strong>Sorun Açıklaması:</strong></p>
                        <div class="alert alert-light">
                            <?= nl2br(h($request['problem_description'])) ?>
                        </div>
                    </div>
                    <?php endif; ?>
                    
                    <?php if ($request['preferred_date'] || $request['preferred_time']): ?>
                    <div class="mt-3">
                        <p><strong>Tercih Edilen Randevu:</strong></p>
                        <p class="text-muted">
                            <?php if ($request['preferred_date']): ?>
                                <i class="fas fa-calendar me-1"></i><?= formatDate($request['preferred_date'], 'd.m.Y') ?>
                            <?php endif; ?>
                            <?php if ($request['preferred_time']): ?>
                                <i class="fas fa-clock me-1 ms-3"></i><?= h($request['preferred_time']) ?>
                            <?php endif; ?>
                        </p>
                    </div>
                    <?php endif; ?>
                    
                    <?php if ($request['additional_notes']): ?>
                    <div class="mt-3">
                        <p><strong>Ek Notlar:</strong></p>
                        <div class="alert alert-light">
                            <?= nl2br(h($request['additional_notes'])) ?>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Hazırlanmış Teklif -->
            <?php if ($quote): ?>
            <div class="nice-card mb-4">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center mb-3">
                        <h5 class="card-title mb-0">
                            <i class="fas fa-file-invoice-dollar me-2"></i>Hazırlanmış Teklif
                        </h5>
                        <span class="nice-badge nice-badge-success">Teklif Hazırlandı</span>
                    </div>
                    
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <p><strong>Teklif Başlığı:</strong> <?= h($quote['title']) ?></p>
                            <p><strong>Geçerlilik Tarihi:</strong> <?= formatDate($quote['valid_until'], 'd.m.Y H:i') ?></p>
                        </div>
                        <div class="col-md-6">
                            <p><strong>Teklif Tarihi:</strong> <?= formatDate($quote['created_at'], 'd.m.Y H:i') ?></p>
                            <p><strong>Durum:</strong> 
                                <span class="nice-badge nice-badge-primary"><?= ucfirst($quote['status']) ?></span>
                            </p>
                        </div>
                    </div>
                    
                    <!-- Teklif Kalemleri -->
                    <?php if (!empty($quote_items)): ?>
                    <h6 class="text-primary mb-3">Teklif Kalemleri</h6>
                    <div class="table-responsive">
                        <table class="nice-table">
                            <thead>
                                <tr>
                                    <th>Ürün/Hizmet</th>
                                    <th>Açıklama</th>
                                    <th>Adet</th>
                                    <th>Birim Fiyat</th>
                                    <th>Toplam</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($quote_items as $item): ?>
                                <tr>
                                    <td><strong><?= h($item['description'] ?? 'Ürün Adı Yok') ?></strong></td>
                                    <td class="text-muted"><?= h($item['description']) ?></td>
                                    <td><?= $item['quantity'] ?></td>
                                    <td><?= formatMoney($item['unit_price']) ?></td>
                                    <td><strong><?= formatMoney($item['line_total']) ?></strong></td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    <?php endif; ?>
                    
                    <!-- Teklif Özeti -->
                    <div class="row mt-4">
                        <div class="col-md-6 offset-md-6">
                            <table class="table table-borderless">
                                <tr>
                                    <td>Ara Toplam:</td>
                                    <td class="text-end"><?= formatMoney($quote['subtotal']) ?></td>
                                </tr>
                                <?php if ($quote['discount_amount'] > 0): ?>
                                <tr>
                                    <td>İndirim (<?= $quote['discount_rate'] ?>%):</td>
                                    <td class="text-end text-danger">-<?= formatMoney($quote['discount_amount']) ?></td>
                                </tr>
                                <?php endif; ?>
                                <tr>
                                    <td>KDV (<?= $quote['tax_rate'] ?>%):</td>
                                    <td class="text-end"><?= formatMoney($quote['tax_amount']) ?></td>
                                </tr>
                                <tr class="border-top">
                                    <td><strong>Toplam:</strong></td>
                                    <td class="text-end"><strong class="text-primary fs-5"><?= formatMoney($quote['total_amount']) ?></strong></td>
                                </tr>
                            </table>
                        </div>
                    </div>
                    
                    <?php if ($quote['notes']): ?>
                    <div class="mt-3">
                        <p><strong>Teklif Notları:</strong></p>
                        <div class="alert alert-info">
                            <?= nl2br(h($quote['notes'])) ?>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
            <?php endif; ?>
        </div>

        <div class="col-lg-4">
            <!-- İşlemler -->
            <div class="nice-card mb-4">
                <div class="card-body">
                    <h5 class="card-title">
                        <i class="fas fa-cogs me-2"></i>İşlemler
                    </h5>
                    
                    <div class="d-grid gap-2">
                        <?php if ($request['status'] === 'NEW' && !$quote): ?>
                        <a href="create_quote.php?request_id=<?= $request['id'] ?>" class="nice-btn nice-btn-primary">
                            <i class="fas fa-plus"></i> Teklif Hazırla
                        </a>
                        <?php endif; ?>
                        
                        <?php if ($quote): ?>
                        <a href="<?= $quote['quote_link'] ?>" target="_blank" class="nice-btn nice-btn-info">
                            <i class="fas fa-external-link-alt"></i> Teklifi Görüntüle
                        </a>
                        
                        <button type="button" class="nice-btn nice-btn-success" onclick="resendQuote(<?= $quote['id'] ?>)">
                            <i class="fas fa-paper-plane"></i> Teklifi Tekrar Gönder
                        </button>
                        <?php endif; ?>
                        
                        <?php if (in_array($request['status'], ['NEW', 'IN_PROGRESS'])): ?>
                        <button type="button" class="nice-btn nice-btn-warning" onclick="updateStatus('IN_PROGRESS')">
                            <i class="fas fa-play"></i> İşleme Al
                        </button>
                        
                        <button type="button" class="nice-btn nice-btn-danger" onclick="cancelRequest()">
                            <i class="fas fa-times"></i> İptal Et
                        </button>
                        <?php endif; ?>
                        
                        <a href="quotes.php" class="nice-btn nice-btn-outline-secondary">
                            <i class="fas fa-arrow-left"></i> Geri Dön
                        </a>
                    </div>
                </div>
            </div>

            <!-- Durum Geçmişi -->
            <div class="nice-card">
                <div class="card-body">
                    <h5 class="card-title">
                        <i class="fas fa-history me-2"></i>Durum Geçmişi
                    </h5>
                    
                    <div class="timeline">
                        <div class="timeline-item">
                            <div class="timeline-marker bg-primary"></div>
                            <div class="timeline-content">
                                <h6 class="timeline-title">Başvuru Alındı</h6>
                                <p class="timeline-text text-muted">
                                    <?= formatDate($request['created_at'], 'd.m.Y H:i') ?>
                                </p>
                            </div>
                        </div>
                        
                        <?php if ($quote): ?>
                        <div class="timeline-item">
                            <div class="timeline-marker bg-success"></div>
                            <div class="timeline-content">
                                <h6 class="timeline-title">Teklif Hazırlandı</h6>
                                <p class="timeline-text text-muted">
                                    <?= formatDate($quote['created_at'], 'd.m.Y H:i') ?>
                                </p>
                            </div>
                        </div>
                        <?php endif; ?>
                        
                        <?php if ($request['status'] === 'CANCELLED'): ?>
                        <div class="timeline-item">
                            <div class="timeline-marker bg-danger"></div>
                            <div class="timeline-content">
                                <h6 class="timeline-title">İptal Edildi</h6>
                                <p class="timeline-text text-muted">
                                    <?= formatDate($request['updated_at'], 'd.m.Y H:i') ?>
                                </p>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

<style>
.timeline {
    position: relative;
    padding-left: 30px;
}

.timeline::before {
    content: '';
    position: absolute;
    left: 15px;
    top: 0;
    bottom: 0;
    width: 2px;
    background: #e9ecef;
}

.timeline-item {
    position: relative;
    margin-bottom: 20px;
}

.timeline-marker {
    position: absolute;
    left: -22px;
    top: 5px;
    width: 12px;
    height: 12px;
    border-radius: 50%;
    border: 2px solid #fff;
    box-shadow: 0 0 0 2px #e9ecef;
}

.timeline-content {
    background: #f8f9fa;
    padding: 15px;
    border-radius: 8px;
    border-left: 3px solid var(--nice-primary);
}

.timeline-title {
    margin: 0 0 5px 0;
    font-size: 14px;
    font-weight: 600;
}

.timeline-text {
    margin: 0;
    font-size: 12px;
}
</style>

<script>
function updateStatus(status) {
    if (confirm('Başvuru durumunu güncellemek istediğinizden emin misiniz?')) {
        fetch('update_quote_status.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                quote_request_id: <?= $quote_request_id ?>,
                status: status
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                location.reload();
            } else {
                alert('Hata: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Bir hata oluştu.');
        });
    }
}

function cancelRequest() {
    if (confirm('Bu başvuruyu iptal etmek istediğinizden emin misiniz?')) {
        updateStatus('CANCELLED');
    }
}

function resendQuote(quoteId) {
    if (confirm('Teklifi tekrar göndermek istediğinizden emin misiniz?')) {
        fetch('resend_quote.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                quote_id: quoteId
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Teklif başarıyla gönderildi!');
            } else {
                alert('Hata: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Bir hata oluştu.');
        });
    }
}
</script>

<?php require_once 'includes/footer.php'; ?>