<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';

if (!isLoggedIn()) {
    redirect('login.php');
}

$success_message = '';
$error_message = '';

// Kategorileri al
try {
    $stmt = $pdo->query("SELECT * FROM product_categories WHERE is_active = 1 ORDER BY name");
    $categories = $stmt->fetchAll();
} catch (Exception $e) {
    $categories = [];
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $category_id = intval($_POST['category_id'] ?? 0);
    $name = trim($_POST['name'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $type = $_POST['type'] ?? '';
    $sku = trim($_POST['sku'] ?? '');
    $price = floatval($_POST['price'] ?? 0);
    $cost_price = floatval($_POST['cost_price'] ?? 0);
    $stock_quantity = intval($_POST['stock_quantity'] ?? 0);
    $min_stock_level = intval($_POST['min_stock_level'] ?? 0);
    $estimated_duration = intval($_POST['estimated_duration'] ?? 0);
    $warranty_period = intval($_POST['warranty_period'] ?? 0);
    $is_active = isset($_POST['is_active']);
    
    $errors = [];
    
    if (empty($name)) $errors[] = 'Ürün/hizmet adı gereklidir.';
    if (empty($type)) $errors[] = 'Tür seçimi gereklidir.';
    if ($price <= 0) $errors[] = 'Geçerli bir fiyat giriniz.';
    
    // SKU kontrolü
    if ($sku) {
        $stmt = $pdo->prepare("SELECT id FROM products WHERE sku = ?");
        $stmt->execute([$sku]);
        if ($stmt->fetch()) {
            $errors[] = 'Bu SKU zaten kullanılıyor.';
        }
    }
    
    if (empty($errors)) {
        try {
            $stmt = $pdo->prepare("
                INSERT INTO products (category_id, name, description, type, sku, price, cost_price, 
                                    stock_quantity, min_stock_level, estimated_duration, warranty_period, is_active) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            
            $stmt->execute([
                $category_id ?: null, $name, $description, $type, $sku ?: null, $price, $cost_price,
                $stock_quantity, $min_stock_level, $estimated_duration, $warranty_period, $is_active ? 1 : 0
            ]);
            
            $product_id = $pdo->lastInsertId();
            
            // Audit log
            logAudit('product_created', 'products', $product_id, null, [
                'name' => $name,
                'type' => $type,
                'price' => $price,
                'created_by' => $_SESSION['user_id']
            ]);
            
            $success_message = 'Ürün başarıyla eklendi.';
            
            // Formu temizle
            $_POST = [];
            
        } catch (Exception $e) {
            error_log("Product add error: " . $e->getMessage());
            $error_message = 'Ürün eklenirken hata oluştu: ' . $e->getMessage();
        }
    } else {
        $error_message = implode('<br>', $errors);
    }
}
?>
<?php require_once 'includes/header.php'; ?>

<section class="section">
    <div class="pagetitle">
        <h1>Ürün Ekle</h1>
        <nav>
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="dashboard.php">Ana Sayfa</a></li>
                <li class="breadcrumb-item"><a href="products.php">Ürünler</a></li>
                <li class="breadcrumb-item active">Ürün Ekle</li>
            </ol>
        </nav>
    </div>

    <?php if ($success_message): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <i class="fas fa-check-circle me-2"></i><?= $success_message ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php endif; ?>

    <?php if ($error_message): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <i class="fas fa-exclamation-triangle me-2"></i><?= $error_message ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php endif; ?>

    <div class="row">
        <div class="col-lg-8">
            <div class="nice-card">
                <div class="card-body">
                    <h5 class="card-title text-primary">
                        <i class="fas fa-plus-circle me-2"></i>Yeni Ürün/Hizmet Bilgileri
                    </h5>

                    <form method="POST" class="needs-validation" novalidate>
                        <div class="row">
                            <!-- Temel Bilgiler -->
                            <div class="col-md-6 mb-3">
                                <label for="name" class="form-label">Ürün/Hizmet Adı <span class="text-danger">*</span></label>
                                <input type="text" class="nice-form-control" id="name" name="name" 
                                       value="<?= h($_POST['name'] ?? '') ?>" required>
                                <div class="invalid-feedback">Ürün/hizmet adı gereklidir.</div>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="category_id" class="form-label">Kategori</label>
                                <select class="nice-form-control" id="category_id" name="category_id">
                                    <option value="">Kategori Seçin</option>
                                    <?php foreach ($categories as $category): ?>
                                    <option value="<?= $category['id'] ?>" 
                                            <?= (($_POST['category_id'] ?? 0) == $category['id']) ? 'selected' : '' ?>>
                                        <?= h($category['name']) ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="type" class="form-label">Tür <span class="text-danger">*</span></label>
                                <select class="nice-form-control" id="type" name="type" required>
                                    <option value="">Tür Seçin</option>
                                    <option value="service" <?= (($_POST['type'] ?? '') == 'service') ? 'selected' : '' ?>>Hizmet</option>
                                    <option value="part" <?= (($_POST['type'] ?? '') == 'part') ? 'selected' : '' ?>>Parça</option>
                                    <option value="package" <?= (($_POST['type'] ?? '') == 'package') ? 'selected' : '' ?>>Paket</option>
                                </select>
                                <div class="invalid-feedback">Tür seçimi gereklidir.</div>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="sku" class="form-label">SKU/Kod</label>
                                <input type="text" class="nice-form-control" id="sku" name="sku" 
                                       value="<?= h($_POST['sku'] ?? '') ?>" placeholder="Ürün kodu (opsiyonel)">
                                <small class="form-text text-muted">Benzersiz ürün kodu</small>
                            </div>

                            <div class="col-12 mb-3">
                                <label for="description" class="form-label">Açıklama</label>
                                <textarea class="nice-form-control" id="description" name="description" rows="4" 
                                          placeholder="Ürün/hizmet açıklaması..."><?= h($_POST['description'] ?? '') ?></textarea>
                            </div>

                            <!-- Fiyat Bilgileri -->
                            <div class="col-12 mb-4">
                                <h6 class="text-primary border-bottom pb-2">
                                    <i class="fas fa-money-bill-wave me-2"></i>Fiyat Bilgileri
                                </h6>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="price" class="form-label">Satış Fiyatı (₺) <span class="text-danger">*</span></label>
                                <input type="number" class="nice-form-control" id="price" name="price" 
                                       value="<?= $_POST['price'] ?? '' ?>" step="0.01" min="0" required>
                                <div class="invalid-feedback">Geçerli bir fiyat giriniz.</div>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="cost_price" class="form-label">Maliyet Fiyatı (₺)</label>
                                <input type="number" class="nice-form-control" id="cost_price" name="cost_price" 
                                       value="<?= $_POST['cost_price'] ?? '' ?>" step="0.01" min="0">
                                <small class="form-text text-muted">İç maliyet hesaplaması için</small>
                            </div>

                            <!-- Stok Bilgileri -->
                            <div class="col-12 mb-4">
                                <h6 class="text-primary border-bottom pb-2">
                                    <i class="fas fa-boxes me-2"></i>Stok Bilgileri
                                </h6>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="stock_quantity" class="form-label">Stok Miktarı</label>
                                <input type="number" class="nice-form-control" id="stock_quantity" name="stock_quantity" 
                                       value="<?= $_POST['stock_quantity'] ?? '0' ?>" min="0">
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="min_stock_level" class="form-label">Minimum Stok Seviyesi</label>
                                <input type="number" class="nice-form-control" id="min_stock_level" name="min_stock_level" 
                                       value="<?= $_POST['min_stock_level'] ?? '0' ?>" min="0">
                                <small class="form-text text-muted">Uyarı için minimum seviye</small>
                            </div>

                            <!-- Hizmet Bilgileri -->
                            <div class="col-12 mb-4">
                                <h6 class="text-primary border-bottom pb-2">
                                    <i class="fas fa-tools me-2"></i>Hizmet Bilgileri
                                </h6>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="estimated_duration" class="form-label">Tahmini Süre (dakika)</label>
                                <input type="number" class="nice-form-control" id="estimated_duration" name="estimated_duration" 
                                       value="<?= $_POST['estimated_duration'] ?? '0' ?>" min="0">
                                <small class="form-text text-muted">Hizmet süresi (dakika cinsinden)</small>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="warranty_period" class="form-label">Garanti Süresi (gün)</label>
                                <input type="number" class="nice-form-control" id="warranty_period" name="warranty_period" 
                                       value="<?= $_POST['warranty_period'] ?? '0' ?>" min="0">
                                <small class="form-text text-muted">Garanti süresi (gün cinsinden)</small>
                            </div>

                            <!-- Durum -->
                            <div class="col-12 mb-4">
                                <div class="form-check form-switch">
                                    <input class="form-check-input" type="checkbox" id="is_active" name="is_active" 
                                           <?= isset($_POST['is_active']) ? 'checked' : 'checked' ?>>
                                    <label class="form-check-label" for="is_active">
                                        <strong>Aktif</strong>
                                        <small class="text-muted d-block">Ürün/hizmet aktif olarak kullanılabilir</small>
                                    </label>
                                </div>
                            </div>
                        </div>

                        <div class="d-flex gap-2">
                            <button type="submit" class="nice-btn nice-btn-success">
                                <i class="fas fa-save me-2"></i>Kaydet
                            </button>
                            <a href="products.php" class="nice-btn nice-btn-outline-secondary">
                                <i class="fas fa-arrow-left me-2"></i>Geri Dön
                            </a>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <!-- Sağ Kolon - Yardım ve Bilgiler -->
        <div class="col-lg-4">
            <div class="nice-card">
                <div class="card-body">
                    <h5 class="card-title text-primary">
                        <i class="fas fa-info-circle me-2"></i>Yardım
                    </h5>
                    
                    <div class="alert alert-info">
                        <h6><i class="fas fa-lightbulb me-2"></i>İpuçları</h6>
                        <ul class="mb-0 small">
                            <li><strong>SKU:</strong> Benzersiz ürün kodu, stok takibi için önemlidir</li>
                            <li><strong>Tür:</strong> Hizmet, Parça veya Paket olarak sınıflandırın</li>
                            <li><strong>Maliyet:</strong> Kar marjı hesaplaması için maliyet fiyatını girin</li>
                            <li><strong>Stok:</strong> Fiziksel ürünler için stok takibi yapılır</li>
                        </ul>
                    </div>

                    <div class="alert alert-warning">
                        <h6><i class="fas fa-exclamation-triangle me-2"></i>Dikkat</h6>
                        <p class="mb-0 small">
                            Ürün eklendikten sonra SKU değiştirilemez. 
                            Dikkatli bir şekilde benzersiz kod belirleyin.
                        </p>
                    </div>
                </div>
            </div>

            <!-- Hızlı İstatistikler -->
            <div class="nice-card">
                <div class="card-body">
                    <h5 class="card-title text-primary">
                        <i class="fas fa-chart-bar me-2"></i>İstatistikler
                    </h5>
                    
                    <?php
                    try {
                        $stats = $pdo->query("
                            SELECT 
                                COUNT(*) as total_products,
                                COUNT(CASE WHEN is_active = 1 THEN 1 END) as active_products,
                                COUNT(CASE WHEN type = 'service' THEN 1 END) as services,
                                COUNT(CASE WHEN type = 'part' THEN 1 END) as parts
                            FROM products
                        ")->fetch();
                    } catch (Exception $e) {
                        $stats = ['total_products' => 0, 'active_products' => 0, 'services' => 0, 'parts' => 0];
                    }
                    ?>
                    
                    <div class="row text-center">
                        <div class="col-6 mb-3">
                            <div class="border rounded p-2">
                                <h4 class="text-primary mb-0"><?= $stats['total_products'] ?></h4>
                                <small class="text-muted">Toplam Ürün</small>
                            </div>
                        </div>
                        <div class="col-6 mb-3">
                            <div class="border rounded p-2">
                                <h4 class="text-success mb-0"><?= $stats['active_products'] ?></h4>
                                <small class="text-muted">Aktif Ürün</small>
                            </div>
                        </div>
                        <div class="col-6">
                            <div class="border rounded p-2">
                                <h4 class="text-info mb-0"><?= $stats['services'] ?></h4>
                                <small class="text-muted">Hizmet</small>
                            </div>
                        </div>
                        <div class="col-6">
                            <div class="border rounded p-2">
                                <h4 class="text-warning mb-0"><?= $stats['parts'] ?></h4>
                                <small class="text-muted">Parça</small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

<script>
// Form validation
(function() {
    'use strict';
    window.addEventListener('load', function() {
        var forms = document.getElementsByClassName('needs-validation');
        var validation = Array.prototype.filter.call(forms, function(form) {
            form.addEventListener('submit', function(event) {
                if (form.checkValidity() === false) {
                    event.preventDefault();
                    event.stopPropagation();
                }
                form.classList.add('was-validated');
            }, false);
        });
    }, false);
})();

// SKU otomatik oluşturma
document.getElementById('name').addEventListener('blur', function() {
    const skuField = document.getElementById('sku');
    if (!skuField.value && this.value) {
        // Basit SKU oluşturma
        const sku = this.value
            .toUpperCase()
            .replace(/[^A-Z0-9]/g, '')
            .substring(0, 8) + 
            Math.floor(Math.random() * 100).toString().padStart(2, '0');
        skuField.value = sku;
    }
});

// Kar marjı hesaplama
function calculateMargin() {
    const price = parseFloat(document.getElementById('price').value) || 0;
    const cost = parseFloat(document.getElementById('cost_price').value) || 0;
    
    if (price > 0 && cost > 0) {
        const margin = ((price - cost) / price * 100).toFixed(2);
        console.log('Kar Marjı: %' + margin);
    }
}

document.getElementById('price').addEventListener('input', calculateMargin);
document.getElementById('cost_price').addEventListener('input', calculateMargin);
</script>

<?php require_once 'includes/footer.php'; ?>