<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'message' => 'Oturum süresi dolmuş.'], 401);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'message' => 'Geçersiz istek metodu.'], 405);
}

try {
    $request_id = intval($_POST['request_id'] ?? 0);
    $products = $_POST['products'] ?? [];
    $manual_items = $_POST['manual_items'] ?? [];
    $discount_percentage = floatval($_POST['discount_percentage'] ?? 0);
    $tax_rate = floatval($_POST['tax_rate'] ?? 18);
    $terms_conditions = trim($_POST['terms_conditions'] ?? '');
    $internal_notes = trim($_POST['internal_notes'] ?? '');
    
    if (!$request_id) {
        jsonResponse(['success' => false, 'message' => 'Geçersiz başvuru ID.']);
    }
    
    // Ürün seçimi veya manuel ürün girişi olmalı
    if (empty($products) && empty($manual_items)) {
        jsonResponse(['success' => false, 'message' => 'En az bir ürün/hizmet seçmeli veya manuel olarak eklemelisiniz.']);
    }
    
    $pdo->beginTransaction();
    
    // Başvuru kontrolü
    $stmt = $pdo->prepare("SELECT id, status FROM quote_requests WHERE id = ?");
    $stmt->execute([$request_id]);
    $request = $stmt->fetch();
    
    if (!$request) {
        throw new Exception('Başvuru bulunamadı.');
    }
    
    // Zaten teklif var mı kontrol et
    $stmt = $pdo->prepare("SELECT id FROM quotes WHERE quote_request_id = ?");
    $stmt->execute([$request_id]);
    if ($stmt->fetch()) {
        throw new Exception('Bu başvuru için zaten teklif oluşturulmuş.');
    }
    
    // Teklif numarası ve token oluştur
    $quote_number = generateQuoteNumber();
    $quote_token = generateToken(32);
    
    // Geçerlilik tarihi (48 saat sonra)
    $valid_until = date('Y-m-d H:i:s', strtotime('+' . QUOTE_EXPIRY_HOURS . ' hours'));
    
    // Teklif oluştur
    $stmt = $pdo->prepare("
        INSERT INTO quotes (quote_request_id, quote_number, quote_token, discount_percentage, 
                           tax_rate, valid_until, terms_conditions, internal_notes, created_by) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
    ");
    $stmt->execute([
        $request_id, $quote_number, $quote_token, $discount_percentage,
        $tax_rate / 100, $valid_until, $terms_conditions, $internal_notes, $_SESSION['user_id']
    ]);
    
    $quote_id = $pdo->lastInsertId();
    
    // Teklif kalemlerini ekle ve toplamları hesapla
    $subtotal = 0;
    $sort_order = 0;
    
    // Seçili ürünleri ekle
    foreach ($products as $product_data) {
        $product_id = intval($product_data['id'] ?? 0);
        $name = trim($product_data['name'] ?? '');
        $quantity = floatval($product_data['quantity'] ?? 1);
        $unit_price = floatval($product_data['price'] ?? 0);
        
        if (!$product_id || !$name || $quantity <= 0 || $unit_price < 0) {
            continue;
        }
        
        $line_total = $quantity * $unit_price;
        $subtotal += $line_total;
        
        $stmt = $pdo->prepare("
            INSERT INTO quote_items (quote_id, product_id, description, quantity, unit_price, 
                                   discount_amount, tax_rate, line_total, sort_order) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $quote_id, $product_id, $name, $quantity, $unit_price,
            0, $tax_rate / 100, $line_total, ++$sort_order
        ]);
    }
    
    // Manuel ürünleri ekle
    foreach ($manual_items as $manual_item) {
        $name = trim($manual_item['name'] ?? '');
        $description = trim($manual_item['description'] ?? '');
        $quantity = floatval($manual_item['quantity'] ?? 1);
        $unit_price = floatval($manual_item['unit_price'] ?? 0);
        
        if (!$name || $quantity <= 0 || $unit_price < 0) {
            continue;
        }
        
        $line_total = $quantity * $unit_price;
        $subtotal += $line_total;
        
        $stmt = $pdo->prepare("
            INSERT INTO quote_items (quote_id, product_id, description, quantity, unit_price, 
                                   discount_amount, tax_rate, line_total, sort_order) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $quote_id, null, $description ?: $name, $quantity, $unit_price,
            0, $tax_rate / 100, $line_total, ++$sort_order
        ]);
    }
    
    // Toplamları hesapla
    $discount_amount = $subtotal * ($discount_percentage / 100);
    $after_discount = $subtotal - $discount_amount;
    $tax_amount = $after_discount * ($tax_rate / 100);
    $total_amount = $after_discount + $tax_amount;
    
    // Teklif toplamlarını güncelle
    $stmt = $pdo->prepare("
        UPDATE quotes 
        SET subtotal = ?, discount_amount = ?, tax_amount = ?, total_amount = ?
        WHERE id = ?
    ");
    $stmt->execute([$subtotal, $discount_amount, $tax_amount, $total_amount, $quote_id]);
    
    // Başvuru durumunu güncelle
    $stmt = $pdo->prepare("UPDATE quote_requests SET status = 'IN_PROGRESS' WHERE id = ?");
    $stmt->execute([$request_id]);
    
    // Audit log
    logAudit('quote_created', 'quotes', $quote_id, null, [
        'quote_number' => $quote_number,
        'request_id' => $request_id,
        'total_amount' => $total_amount
    ]);
    
    $pdo->commit();
    
    // Teklif linkini oluştur ve durumu SENT yap
    $quote_token = bin2hex(random_bytes(32));
    $quote_link = (isset($_SERVER['HTTPS']) ? 'https' : 'http') . "://" . $_SERVER['HTTP_HOST'] . "/view_quote.php?token=" . $quote_token;
    
    $stmt = $pdo->prepare("UPDATE quotes SET quote_token = ?, quote_link = ?, status = 'SENT', sent_at = NOW() WHERE id = ?");
    $stmt->execute([$quote_token, $quote_link, $quote_id]);
    
    // PDF oluştur ve e-posta fonksiyonlarını dahil et
    require_once 'quote_functions.php';
    require_once '../includes/email_functions.php';
    $pdf_filename = generateQuotePDF($quote_id, $pdo);
    
    // Teklif bilgilerini al (customer_id için)
    $stmt = $pdo->prepare("
        SELECT q.*, qr.customer_id, qr.vehicle_id 
        FROM quotes q 
        JOIN quote_requests qr ON q.quote_request_id = qr.id 
        WHERE q.id = ?
    ");
    $stmt->execute([$quote_id]);
    $quote_info = $stmt->fetch();
    
    if (!$quote_info) {
        throw new Exception('Teklif bulunamadı.');
    }
    
    // Müşteri bilgilerini al
    $stmt = $pdo->prepare("SELECT * FROM customers WHERE id = ?");
    $stmt->execute([$quote_info['customer_id']]);
    $customer = $stmt->fetch();
    
    if (!$customer) {
        throw new Exception('Müşteri bulunamadı.');
    }
    
    // Müşteriye gerçek e-posta gönder
    try {
        $notification_sent = sendQuoteNotification($customer, $quote_link, $pdf_filename, $quote_info);
    } catch (Exception $e) {
        error_log("Email notification error: " . $e->getMessage());
        $notification_sent = false;
    }
    
    // Başarı mesajını e-posta durumuna göre ayarla
    $message = 'Teklif başarıyla oluşturuldu.';
    if ($notification_sent) {
        $message .= ' E-posta müşteriye gönderildi.';
    } else {
        $message .= ' E-posta gönderilemedi, lütfen manuel olarak müşteriyle iletişime geçin.';
    }
    
    jsonResponse([
        'success' => true,
        'message' => $message,
        'quote_id' => $quote_id,
        'quote_number' => $quote_number,
        'quote_link' => $quote_link,
        'pdf_generated' => !empty($pdf_filename),
        'notification_sent' => $notification_sent
    ]);
    
} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    error_log("Create quote error: " . $e->getMessage());
    jsonResponse(['success' => false, 'message' => $e->getMessage()], 500);
}

?>
