<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    echo json_encode(['success' => false, 'message' => 'Oturum geçersiz.']);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Geçersiz istek metodu.']);
    exit;
}

try {
    $input = json_decode(file_get_contents('php://input'), true);
    $quote_id = intval($input['quote_id'] ?? 0);
    
    if ($quote_id <= 0) {
        throw new Exception('Geçersiz teklif ID.');
    }
    
    // Teklif bilgilerini al
    $stmt = $pdo->prepare("
        SELECT q.*, qr.id as request_id, qr.status as request_status,
               c.first_name, c.last_name
        FROM quotes q
        JOIN quote_requests qr ON q.quote_request_id = qr.id
        JOIN customers c ON qr.customer_id = c.id
        WHERE q.id = ?
    ");
    $stmt->execute([$quote_id]);
    $quote = $stmt->fetch();
    
    if (!$quote) {
        throw new Exception('Teklif bulunamadı.');
    }
    
    // Silinebilir durumda mı kontrol et
    $deletable_statuses = ['DRAFT', 'SENT', 'VIEWED'];
    if (!in_array($quote['status'], $deletable_statuses)) {
        throw new Exception('Bu durumda olan teklif silinemez. Sadece taslak, gönderilmiş veya görüntülenmiş teklifler silinebilir.');
    }
    
    // Ödeme yapılmış mı kontrol et (orders üzerinden)
    $stmt = $pdo->prepare("
        SELECT p.id 
        FROM payments p 
        JOIN orders o ON p.order_id = o.id 
        WHERE o.quote_id = ? AND p.status = 'completed'
    ");
    $stmt->execute([$quote_id]);
    if ($stmt->fetch()) {
        throw new Exception('Ödeme yapılmış teklif silinemez.');
    }
    
    $pdo->beginTransaction();
    
    // Teklif kalemlerini sil
    $stmt = $pdo->prepare("DELETE FROM quote_items WHERE quote_id = ?");
    $stmt->execute([$quote_id]);
    
    // Teklifi sil
    $stmt = $pdo->prepare("DELETE FROM quotes WHERE id = ?");
    $stmt->execute([$quote_id]);
    
    // Başvuru durumunu güncelle (eğer başka teklif yoksa)
    $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM quotes WHERE quote_request_id = ?");
    $stmt->execute([$quote['request_id']]);
    $remaining_quotes = $stmt->fetch()['count'];
    
    if ($remaining_quotes == 0) {
        // Başka teklif yoksa durumu NEW yap
        $stmt = $pdo->prepare("UPDATE quote_requests SET status = 'NEW' WHERE id = ?");
        $stmt->execute([$quote['request_id']]);
    }
    
    // Audit log
    logAudit('quote_deleted', 'quotes', $quote_id, null, [
        'quote_number' => $quote['quote_number'],
        'customer_name' => $quote['first_name'] . ' ' . $quote['last_name'],
        'deleted_by' => $_SESSION['user_id']
    ]);
    
    $pdo->commit();
    
    echo json_encode([
        'success' => true,
        'message' => 'Teklif başarıyla silindi.',
        'quote_number' => $quote['quote_number']
    ]);
    
} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    error_log("Delete quote error: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
?>
