<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/appointment_functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Oturum geçersiz.']);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Geçersiz istek metodu.']);
    exit;
}

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!isset($input['action'])) {
        throw new Exception('İşlem belirtilmedi.');
    }
    
    $action = $input['action'];
    $appointmentId = intval($input['appointment_id'] ?? 0);
    
    if (!$appointmentId) {
        throw new Exception('Geçersiz randevu ID.');
    }
    
    switch ($action) {
        case 'update_status':
            if (!isset($input['status'])) {
                throw new Exception('Durum belirtilmedi.');
            }
            
            $status = $input['status'];
            $validStatuses = ['PENDING', 'CONFIRMED', 'IN_PROGRESS', 'COMPLETED', 'CANCELLED', 'NO_SHOW'];
            
            if (!in_array($status, $validStatuses)) {
                throw new Exception('Geçersiz durum.');
            }
            
            $adminNotes = $input['admin_notes'] ?? null;
            
            if (updateAppointmentStatus($appointmentId, $status, $adminNotes)) {
                // Audit log
                logAudit('appointment_status_updated', 'appointments', $appointmentId, $_SESSION['user_id'], [
                    'old_status' => 'unknown', // Eski durumu almak için ek sorgu gerekir
                    'new_status' => $status,
                    'admin_notes' => $adminNotes
                ]);
                
                echo json_encode([
                    'success' => true,
                    'message' => 'Randevu durumu güncellendi.'
                ]);
            } else {
                throw new Exception('Randevu durumu güncellenemedi.');
            }
            break;
            
        case 'send_reminder':
            if (sendAppointmentReminder($appointmentId)) {
                // Audit log
                logAudit('appointment_reminder_sent', 'appointments', $appointmentId, $_SESSION['user_id'], [
                    'sent_by' => $_SESSION['user_id']
                ]);
                
                echo json_encode([
                    'success' => true,
                    'message' => 'Hatırlatma başarıyla gönderildi.'
                ]);
            } else {
                throw new Exception('Hatırlatma gönderilemedi.');
            }
            break;
            
        case 'assign_technician':
            if (!isset($input['technician_id'])) {
                throw new Exception('Teknisyen ID belirtilmedi.');
            }
            
            $technicianId = intval($input['technician_id']);
            
            $stmt = $pdo->prepare("
                UPDATE appointments 
                SET assigned_technician_id = ?, updated_at = NOW() 
                WHERE id = ?
            ");
            
            if ($stmt->execute([$technicianId, $appointmentId])) {
                // Audit log
                logAudit('appointment_technician_assigned', 'appointments', $appointmentId, $_SESSION['user_id'], [
                    'technician_id' => $technicianId,
                    'assigned_by' => $_SESSION['user_id']
                ]);
                
                echo json_encode([
                    'success' => true,
                    'message' => 'Teknisyen atandı.'
                ]);
            } else {
                throw new Exception('Teknisyen atanamadı.');
            }
            break;
            
        case 'delete':
            // Sadece PENDING durumundaki randevular silinebilir
            $stmt = $pdo->prepare("SELECT status FROM appointments WHERE id = ?");
            $stmt->execute([$appointmentId]);
            $appointment = $stmt->fetch();
            
            if (!$appointment) {
                throw new Exception('Randevu bulunamadı.');
            }
            
            if ($appointment['status'] !== 'PENDING') {
                throw new Exception('Sadece beklemede olan randevular silinebilir.');
            }
            
            $stmt = $pdo->prepare("DELETE FROM appointments WHERE id = ?");
            
            if ($stmt->execute([$appointmentId])) {
                // Audit log
                logAudit('appointment_deleted', 'appointments', $appointmentId, $_SESSION['user_id'], [
                    'deleted_by' => $_SESSION['user_id']
                ]);
                
                echo json_encode([
                    'success' => true,
                    'message' => 'Randevu silindi.'
                ]);
            } else {
                throw new Exception('Randevu silinemedi.');
            }
            break;
            
        default:
            throw new Exception('Geçersiz işlem.');
    }
    
} catch (Exception $e) {
    error_log("Appointment action error: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
?>
