<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    echo json_encode(['success' => false, 'message' => 'Oturum geçersiz.']);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST' || !isset($_POST['action']) || $_POST['action'] !== 'add_product') {
    echo json_encode(['success' => false, 'message' => 'Geçersiz istek.']);
    exit;
}

try {
    $pdo->beginTransaction();
    
    // Yeni parametreler
    $category_id = intval($_POST['category_id'] ?? 0);
    $category_name = trim($_POST['category_name'] ?? '');
    $name = trim($_POST['name'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $price = floatval($_POST['price'] ?? 0);
    $type = $_POST['type'] ?? 'service';
    
    // Validasyon
    if (empty($name)) {
        throw new Exception('Ürün/hizmet adı gereklidir.');
    }
    
    if ($price <= 0) {
        throw new Exception('Geçerli bir fiyat giriniz.');
    }
    
    // Kategori işlemi
    if ($category_id > 0) {
        // Mevcut kategori seçildi
        $stmt = $pdo->prepare("SELECT id, name FROM product_categories WHERE id = ?");
        $stmt->execute([$category_id]);
        $category = $stmt->fetch();
        
        if (!$category) {
            throw new Exception('Seçilen kategori bulunamadı.');
        }
        
        $final_category_id = $category['id'];
        $final_category_name = $category['name'];
        
    } elseif (!empty($category_name)) {
        // Yeni kategori oluştur
        $stmt = $pdo->prepare("SELECT id FROM product_categories WHERE name = ?");
        $stmt->execute([$category_name]);
        $existing_category = $stmt->fetch();
        
        if ($existing_category) {
            $final_category_id = $existing_category['id'];
            $final_category_name = $category_name;
        } else {
            // Yeni kategori oluştur
            $stmt = $pdo->prepare("INSERT INTO product_categories (name, is_active) VALUES (?, 1)");
            $stmt->execute([$category_name]);
            $final_category_id = $pdo->lastInsertId();
            $final_category_name = $category_name;
            
            logAudit('category_created', 'product_categories', $final_category_id, null, [
                'name' => $category_name,
                'created_by' => $_SESSION['user_id']
            ]);
        }
    } else {
        // Kategori yok - null olarak ekle
        $final_category_id = null;
        $final_category_name = 'Kategori Yok';
    }
    
    // Ürün adı kontrolü (aynı kategoride)
    if ($final_category_id) {
        $stmt = $pdo->prepare("SELECT id FROM products WHERE category_id = ? AND name = ?");
        $stmt->execute([$final_category_id, $name]);
        if ($stmt->fetch()) {
            throw new Exception('Bu kategoride aynı isimde ürün zaten mevcut.');
        }
    }
    
    // Ürünü ekle
    $stmt = $pdo->prepare("
        INSERT INTO products (category_id, name, description, type, price, is_active, created_at) 
        VALUES (?, ?, ?, ?, ?, 1, NOW())
    ");
    $stmt->execute([$final_category_id, $name, $description, $type, $price]);
    
    $product_id = $pdo->lastInsertId();
    
    logAudit('product_created', 'products', $product_id, null, [
        'name' => $name,
        'description' => $description,
        'category_id' => $final_category_id,
        'type' => $type,
        'price' => $price,
        'created_by' => $_SESSION['user_id']
    ]);
    
    $pdo->commit();
    
    echo json_encode([
        'success' => true,
        'message' => 'Ürün başarıyla eklendi.',
        'product_id' => $product_id,
        'category_id' => $final_category_id,
        'category_name' => $final_category_name
    ]);
    
} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    error_log("Add product AJAX error: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
?>
